
/*CoaCC.c ver 1.0*/

/*Overview:
The program CoaCC.c simulates a case-control study using a 
coalescent framework. It assumes a haploid sample of cases 
and a second haploid sample of controls. Of these two samples 
the genealogy is generated, dependent on the user-specified population history.
From this genealogy a distribution of marker-haplotypes 
is generated by allowing for marker-mutation and recombinations 
between marker and gene as well as between markers. The difference 
between the allelic distributions in the case and the control-sample 
(LD) is calculated as d-value.  
The simulation is repeated several times to get the distribution 
of d-values, as a function of  the simulation conditions.

The marker map can be configured, so that it simulates both flanks 
of the disease locus. It is possible to place  up to 5 markers at each 
side of the disease locus. Those markers are numbered

   5 - 4 - 3 - 2 - 1 - 0 - 1 - 2 - 3 - 4 - 5
Flank 2 	disease locus		Flank 1

As long as option -L is not set, the alleles of each marker are set to have 
identical frequencies and all markers are in mutual linkage equlibrium.
		
The simulation is described in detail in S.Zoellner and A. Von Haeseler,
"A coalescent approach to study linkage disequilibrium between SNPs". 
It appeared in Am.J.Hum.Genet., 66:615-628,2000. Please cite this paper 
if You are using the program in a publication.
 
Technical detail:
This C-program has be written by Sebastian Zoellner
(zoellner@eva.mpg.de) for a LINUX system, distribution Red
Hat 6.0. If You have any questions, contact me.

It can be compiled using a cc CoaCC.c -lm -o program.name
command.

*/

#undef DEBUGc 			/*prints intermediate results during tree sim*/   
#undef DEBUGm	  		/*prints intermediate results during foreward simulation*/
#undef DEBUGs

#include <math.h>
#include <time.h>
#include <stdlib.h>
#include <stdio.h>

#define MAXSAMPLE 1001		/*Maximum number of real cases and real controls in the sample*/
#define MAXALLEL 2          	/*Maximum Number of Alleles*/
#define MARKER 5            	/*Number of MARKERs*/
#define MAXWID 10000          	/*Maximum Number of repeats*/ 
#define HAPLOTYPES 32      	/*pow(MAXALLEL,MARKER)*/
#define FLANKS 2		/*Maximum Number of independant SNP-areas flanking the mut locus*/

void usage(char * pn);
float peninput(char type);
void popuinput(char mode);
int sasizeinput(char type);
void recomb(char side);			/*does recombination, changing hap[]*/
void drawhap(char side,int,int);		/*draws a subhaplotype */	
double prob(char side, char[MARKER]);	/*gives frequency of haplotype at side*/
void eqdist(void);			/*assigns allelefreq for eq. frequent alleles in LE*/	
int Ereignis (int , int, double, float);/*Generates the next event in the coalescent*/
double WaTime(int w,int m,double mu,float t); /*Generates the waiting time to the next event*/
void haplofrequinput(void);
void recinput(void);
double mutinput(short side,short mar);


float ran1(long *idum); /*generating random float between 0 and 1*/
double pop(float t);	/*calculates population size at time t*/
short test(double);	/*tests against prob */


 /*Variables for pop(t)*/

float rho;            	   /*rho of paper =>factor to determin starting population size*/
float tau;		   /*tau of paper => start of expansion*/
double N0;                 /*N(0) of paper =>Populationsize in present */

float preva;               /*f of paper => preva*wild=muta*/
						      
long seed;

short flanks,marker;

char hap[MARKER]; 	/*haplotype*/
float rec[FLANKS][MARKER];	/*Prob. of recombination between i-1 and i*/
char newhap[MARKER+1];
double freq[FLANKS][HAPLOTYPES];
char Allele[FLANKS][MARKER+1];

main (int argc, char * argv[])
{

  /*		Deklaration of variables		*/
  /*		""""""""""""""""""""""""		*/

  char E[MAXSAMPLE+MAXSAMPLE+MAXSAMPLE];          /* Mutation- and Coalescent events*/
  float T[MAXSAMPLE+MAXSAMPLE+MAXSAMPLE];          /* Times necessary for each event*/
  int ez;                 /*Counter for Number of events*/
  
  char MsW[FLANKS][MARKER][MAXSAMPLE];        /*State of MARKER of the  Wildtype-chromosome i of {1,...,MAXSAMPLE}*/
  char MsM[FLANKS][MARKER][MAXSAMPLE];         /*State of marker of the  Case-chromosome i of {1,...,MAXSAMPLE}*/
  
  int w,m;              /*Counter for Number of  Lines of Wiltyps/Cases*/
  float tim;	          /*Elapsed time  since the present*/
 
  
  double MutMs[FLANKS][MARKER];           /*Prob of mutation of marker*/
  
  int i,j,k,z,l,a,b,c;
  int S;
 
  double proba;		/*rate of gene-mutation*/
 
  int mut;
                            
  int muta,wild;    	/*nD and nW of paper => Number of cases and wildtypes in sample*/
  
  float pen1;	 	/*Frequency of phenotypical wildtypes that have the ancestry of a case*/ 
  float pen2;       	/*Frequency of phenotypical cases that have the ancestry of a wildtype*/ 
  float zuwa;       	/*Zuwanderung aus einer unendlich grossen Population*/

  int z1;
  short ma[MAXSAMPLE],wa[MAXSAMPLE];/*Variables for chromosome-switching for incomplete penetrance*/
  int Sum;
  int NIMarker;		/*Counter for uninformative Simulations*/
  

  int AnzW[FLANKS][MARKER][MAXALLEL];          /*Number of wiltypes with MS-Allel i*/ 
  int AnzM[FLANKS][MARKER][MAXALLEL];           /*Number of Cases with MS-Allel i*/ 

  int AnzHW[FLANKS][MARKER][FLANKS][MARKER][MAXALLEL][MAXALLEL];          
  /*Number of wiltypes with Haplotypes ij at markers ab*/ 
  int AnzHM[FLANKS][MARKER][FLANKS][MARKER][MAXALLEL][MAXALLEL];           

  float hm;           /*Frequency of Allel i among Cases*/
  float hw;           /*Frequency of Allel i among Wildtypes*/
  float d[FLANKS][MARKER][MAXWID];	/*d-value of marker in repeat maxwid*/
  float dh[FLANKS][MARKER][FLANKS][MARKER][MAXWID]; /*d-value of haplotype in repeat*/
 
  float Schnitt[FLANKS][MARKER]; /*Average d-value of marker*/ 
  float SchnittH[FLANKS][MARKER][FLANKS][MARKER];/*Average d-value of haplotype*/ 

  float Var;	/*Emprical Variance of d-value*/
  float sigma;	/*empirical standard deviation*/
  int Wiederholungen;/*Number of repeats*/


  /*Variables that describe the distribution of d-values*/
  const short steps=50; 	/*Number of categories between 0 and 1, the d-value is counted into*/
  int Co[2*steps];      	/*Number of Simulations that have d in a specific band*/
 
  unsigned int AZeit;
 
  int Test1;
 
  FILE *af;		/*File for the results*/
  FILE *ml;		/*File for the errors appearing*/

  /*flags */
  char edLE;		/*Flag for simple distribution of haplotypes*/
  char hapaus;		/*Flag for printing the haplotypes in file*/
  char habmar;		/*Flag for generating LD of haplotype markers*/
  char maxmar;	       	/*Flag for analyzing the maximal marker*/
  
  float maxd[MAXWID];	/*largest d-value of all markers*/
  int maz[FLANKS][MARKER];	/*couter for marker with largest d-value*/
  float MaAv;		/*average of maximal markers*/

  char side,side2;		/*counter for the number of sides of the marker*/
  char ps=0,rs=0,ms=0;
  char * programm_name =argv[0];
  char * outfile;
  char ha[MARKER+1];
  int HZ;
  short silent =0;

  /*setting of default values*/
  habmar='n';
  edLE='y';
  hapaus='n';
  maxmar='n';

  proba=1e-6;
  muta=200;
  wild =200;
  Wiederholungen=1000;

  pen1 = 0.0;
  pen2 = 0.0;
  outfile="out_file";
  flanks =1;
  marker=1;

  for (side=0;side<FLANKS;++side)
    for (i=0;i<MARKER;++i)
      {
	Allele[side][i]=2;
	rec[side][i]=1e-4;
	MutMs[side][i]=1e-6;
      }
	
  rec[0][0]=5e-5;
  rec[1][0]=5e-5;


  /*			Input			*/
  /*			"""""			*/


 

  while((argc>1) && argv[1][0]=='-')
    {
      switch (argv[1][1])
	{
	case 'b':
	  maxmar='y';
	  break;
	case 'd':

	  hapaus='y';
	
	  break;
	case 'p': /*Option for population*/
	  popuinput(argv[1][2]);
	  ps=1;
	  break;
	case 's':
	  wild=sasizeinput('w');
	  muta=sasizeinput('m');
	  break;
	case 'a':
	  habmar='y';
	  break;
	case 'm':
	  proba=atof(&argv[1][2]);
	  break;
	case 'M':
	  if(argv[1][2]!=0)
	    {
	      if(atof(&argv[1][2])>1.0 || atof(&argv[1][2])<0)
		{
		  fprintf(stderr,"\ninput error: mutation probability <0 or >1");
		  ms=1;
		}
	      else
		for (side=0;side<FLANKS;++side)
		  for (i=0;i<MARKER;++i)
		    MutMs[side][i]=atof(&argv[1][2]);
	    }
	  else
	    ms=1;
	  break;


	case 'l':
	  Wiederholungen = atoi(&argv[1][2]);
	  if(Wiederholungen>MAXWID) 
	    {
	      printf("\ninput error: to many loops. default value used\n");
	      Wiederholungen=1000;
	    }
	  break;
	case 'c':
	  pen1=peninput('a');
	  pen2=peninput('o');
	  break;
	case 'o':
	  outfile=(&argv[1][2]);
	  break;
	case 'L':
	  edLE='n';
	  break;
	case 'H':
	  usage(programm_name);
	  break;
	case 'h':
	  usage(programm_name);
	  break;
	case 'f':
	  flanks=atoi(&argv[1][2]);
	  if (flanks>FLANKS || flanks <1)
	    {
	      fprintf(stderr,"\ninput error: flanks must be 1 or 2. default value used\n");
	      flanks =1;
	    }
	  break;
	case 'n':
	  marker=atoi(&argv[1][2]);
	  if (marker>MARKER || marker <1)
	    {
	      fprintf(stderr,"\ninput error: marker must be between 1 and %d. default value used\n",MARKER);
	      marker =1;
	    }
	  break;
	case 'e':
	  for (side=0;side<FLANKS;++side)
	    {
	      for (i=1;i<MARKER;++i)
		rec[side][i]=atof(&argv[1][2]);
	      rec[side][0]=atof(&argv[1][2])/2;
	    }
	  break;
	case 'r':
	  rs=1;
	  break;
	case 'S':
	  silent=1;
	  break;
	default:
	  fprintf(stderr,"\nbad option %s",argv[1]);
	  usage(programm_name);
	  break;
	}
      --argc;
      ++argv;
    }
  if (ps==0) 
    popuinput(0);
  if (rs==1)
    recinput();
  if(ms==1)
    for (side=0;side<flanks;++side)
      for (i=0;i<marker;++i)
	MutMs[side][i]=mutinput(side,i);
  /*If Flag is set, haplotypes are equally distributed*/
  if (edLE=='y'||edLE=='Y')
    eqdist(); 
  else
    haplofrequinput();

 
    
 
  seed=time(NULL);
  seed=seed*(-1);
  
 

  for (side=0;side<flanks;++side)
    Allele[side][marker]=0;

 
  for (side=0;side<flanks;++side)
    for(i=0;i< marker;++i)
      if(Allele[side][i]>MAXALLEL) 
	{
	  printf("\nInput Error. To many alleles at side %d,marker %d!!\n",side,i);
	  exit(8);
	}
  
 
  /*Parameters are written to file*/
  af=fopen(outfile,"w"); 
  
  fprintf(af,"\nsample size of cases: \t%d",muta);
  fprintf(af,"\nsample size of controls: \t%d\n",wild);
  

  fprintf(af,"\nrate of gene mutation: \t%.2e\n",proba);
  fprintf(af,"\nsize of control population in present: \t%.0e",N0);
  if (rho!=1.0)
    {
      fprintf(af,"\nstarting time of exp. growth in generations:\t%.1f",tau); 
      fprintf(af,"\nstarting size of control population:\t%.0f",N0/rho); 
    }
  else fprintf(af,"\nconstant population size");

  fprintf(af,"\nfrequency of mutation als part of controls: %.2f\n",preva);

  if (pen1==0.0 && pen2 ==0.0)
    fprintf(af,"\nsimple genptype-phenotype interaction"); 
  else
    {
      fprintf(af,"\ncomplex genptype-phenotype interaction:");
      fprintf(af,"\nfrequency of phenotypical cases that have the ancestry of a control:\t%.2f",pen2);
      fprintf(af,"\nfrequency of phenotypical controls that have the ancestry of a case\t%.2f",pen1);
    }   
  for (side=0;side<flanks;++side)
    {
      fprintf(af,"\n\nFlank %d",side+1); 
      for (i=0;i<marker;++i)
	{
	  fprintf(af,"\nNumber of alleles at Marker %d: \t\t%d",i+1,Allele[side][i]); 
	  fprintf(af,"\nPrb. of Marker-Mutation: \t\t%.2e",MutMs[side][i]);
	  fprintf(af,"\nrecombination prb between %d and %d: \t%.2e",i,i+1,rec[side][i]);
	  fprintf(af,"\n\tequvalent to %.1f kb",rec[side][i]*1e5);
	}
    }
 
 
  if (edLE=='y'||edLE=='Y')
    {
      fprintf(af,"\n\nMarker alleles are equally distributed"); 
      fprintf(af,"\nand Markers are in linkage equilibrium");
    } 
  else
    {
      fprintf(af,"\n\nDistribution of marker haplotypes:");
      for(side=0;side<flanks;++side)
	{
	  for(i=0;i<=marker;++i)
	    ha[i]=0;
	  while (ha[marker]==0)
	    {
	     
	      HZ=0;
	      for(i=0;i<marker;++i)
		HZ+=pow(MAXALLEL,i)*ha[i];
	      fprintf(af,"\nfrequency of flank %d haplotype ",side+1);
	      for(i=0;i<marker;++i)
		fprintf(af,"%d",ha[i]);

	      fprintf(af,"\t%f",freq[side][HZ]);

	      ha[0]++;
	      for(i=0;i<marker;++i)
		if (ha[i]==Allele[side][i])
		  {
		    ha[i]=0;
		    ++ha[i+1];
		  }
	    }
	}


      

    }
  fprintf(af,"\n\nSimulation is repeated %d times",Wiederholungen);
  fprintf(af,"\n\nResults:");
  fprintf(af,"\n********");

  fclose(af);
  
  
 



  /* here starts the simulation*/
 



     
   
  /*setting some counters*/  
  NIMarker=0;
  AZeit=0;    
  for (side=0;side<flanks;++side)
    for(a=0;a<marker;++a)
      {	
	 
	Schnitt[side][a]=0;
	for (side2=0;side2<flanks;++side2)
	  for(b=0;b<marker;++b) SchnittH[side][a][side2][b]=0;
      }

       
  for (z=0;z<Wiederholungen;++z)
    {
      /* 		Generating the tree		*/
      /*		"""""""""""""""""""		*/		


      if (AZeit==0) AZeit=time(NULL);    
	  
      /*
	w=(1-zuwa)*w;
	m=(1-zuwa)*m; Zuerst werden die, die spaeter zuwandern entfernt*/

	 

      /*The number of simulated control- and casenumbers are fitted so that the 
	final result consists of wild controls and muta cases*/

      w=wild-pen1*wild+pen2*muta;
      m=muta-pen2*muta+pen1*wild;
	  
	  
	 

      ez=0;
      tim=0.0;
      while (w>1||m>0)
	    
	{

	  ez+=1;

	  T[ez]=WaTime(w,m,proba,tim);

	  tim+=T[ez];

	  E[ez]=Ereignis(w,m,proba,tim);
	

	  if (E[ez]==1) /*Coalescens in Control Population*/  
	    {			
	
#ifdef DEBUGc
	      printf("\nControl Coalescens as event \t%d, time %.2f, cases %d, controls %d",ez,T[ez],m,w);
#endif
	      w-=1;
	    } 
	  if (E[ez]==2) /*Coalescens in Case Population*/  
	    {
#ifdef DEBUGc
	      printf("\nCase Coalescens as event \t%d, time %.2f ,cases %d, controls %d",ez,T[ez],m,w);
#endif
	      m-=1; 
	    }
	      
	  if (E[ez]==3) /*Mutation*/  
	    {
#ifdef DEBUGc
	      printf("\nmutation as event \t\t%d, time %.2f,  cases %d, controls %d",ez,T[ez],m,w);
#endif
	      m-=1;
	      w+=1;
	    }
#ifdef DEBUGc
	  printf("\nZeitpunkt: %.2f\tEreignis: %d \tZeit:%.2f\tPopulationsgroesse: %.0f",tim,E[ez],T[ez],pop(tim)); 
#endif
	}
	  
      /*Rounding of  T[i] to integers*/
      for(i=1;i<ez;++i)
	{
	  S=(int)T[i];
	  T[i+1]+=(T[i]-S);     
	}

#ifdef DEBUGc
      printf("\nEnd of Coalescense-Process, %d events\n",ez);
#endif
	  
	    
	  
	 

      /*Evolution of the markers*/
      /*""""""""""""""""""""""""*/	 


      /*Random determination of the first marker-haplotype*/

      for(side=0;side<flanks;++side)
	{
	  drawhap(side,0,marker);
	  
	  for(a=0;a<marker;++a)
	    MsW[side][a][1]=newhap[a]; 
	}
	  
      w=1;
      m=0;
      for (i=ez;i>=1;--i)
	{
	  if (E[i]==1)
	    /*Coalescent in control, ie one line in control-pop doubles*/
	    {
	      mut=w*ran1(&seed)+1;	/*randomly determining the doubling line*/
		  
	      w+=1;
	      for(side=0;side<flanks;++side)
		for(a=0;a<marker;++a)/*markers in new line have same states 
				       as the one that spawned it*/
		  MsW[side][a][w]=MsW[side][a][mut];
		 
	    }
	      
	  if (E[i]==2) 
	    /*Coalescent in case, ie one line in case-pop doubles*/
		
		
	    {
	      mut=m*ran1(&seed)+1;/*randomly determining the doubling line*/
		  
	      m+=1;
	      for(side=0;side<flanks;++side)
		for(a=0;a<marker;++a) /*markers in new line have same states 
					as the one that spawned it*/
		  MsM[side][a][m]=MsM[side][a][mut];
		 
	    }
      
	  if (E[i]==3) 
	    /*Mutation changes one control to a case*/
		
	    {
	      mut=w*ran1(&seed)+1;/*randomly determining the changing line*/
		  
	      m+=1;
	      for(side=0;side<flanks;++side)
		for(a=0;a<marker;++a)
		  {
		      
		    MsM[side][a][m]=MsW[side][a][mut]; /*new case gets maker states of old control*/
		    for (j=mut;j<w;++j) 
		      MsW[side][a][j]=MsW[side][a][j+1];
		    /*all controls with a count>mut get their marker 
		      states copied to the wildt. of one count less*/ 
		  }
		    
	      w-=1; /*last control is removed*/
	    } 
	      


	  /*In every generation until the next tree-event each chromosome is checked for 
	    recombination and mutation*/
	  for(j=1;j<=T[i];++j)
	    for(side=0;side<flanks;++side)
	      {
		  

		/*First all control chromosomes are checked*/

		for (k=1;k<=w;++k) /*changes of markers in the control-population*/
		  {

		    for(a=0;a<marker;++a)
		      if (test(MutMs[side][a])==0) /*Mutation of marker a*/
			  
			/*Simple one-step model that also fits SNP-mutation*/
			{
#ifdef DEBUGm
			  printf("\nmutation in control %d  marker %d on flank %d",k,a,side);
			  printf("\nchange from %d ",MsW[side][a][k]); 
#endif
			  S=2*ran1(&seed)+1; /*random number of {1,2}*/

			  if (S==1) 
			    if (MsW[side][a][k]<Allele[side][a]-1) MsW[side][a][k]+=1;
			    else MsW[side][a][k]-=1;
			
			  if (S==2) 
			    if( MsW[side][a][k]>0) MsW[side][a][k]-=1; 
			    else MsW[side][a][k]+=1;
#ifdef DEBUGm
			  printf("to %d",MsW[side][a][k]);
#endif
			} 

	      
		      
		    /*A Cross-Over assigns a random marker-allel/-haplotype*/
		      
		    for(a=0;a<marker;++a)
		      hap[a]=MsW[side][a][k];
		      
		    recomb(side); 
		      
		    for(a=0;a<marker;++a)
		      MsW[side][a][k]=hap[a];
		  }
		  

		/*Same simulation for the cases*/

		for (k=1;k<=m;++k)
		  {
		      

		    /*Mutation of Markers*/

		    for(a=0;a<marker;++a)

		      if (test(MutMs[side][a])==0) /*One-step Mutation des MS*/
			{
#ifdef DEBUGm
			  printf("\nmutation in case %d  marker %d on flank %d",k,a,side);
#endif
			  S=2*ran1(&seed)+1;

			  if (S==1)
			    if(MsM[side][a][k]<Allele[side][a]-1) MsM[side][a][k]+=1;
			    else MsM[side][a][k]-=1;

			  if (S==2)
			    if (MsM[side][a][k]>0) MsM[side][a][k]-=1;
			    else MsM[side][a][k]+=1;
			} 
		      
		      
		    /*recombination*/

		    for(a=0;a<marker;++a)
		      hap[a]=MsM[side][a][k];
		      
		    recomb(side);
#ifdef DEBUGm
		
		    printf("\nrecomb with ");
		    for(a=0;a<marker;++a)
		      printf("%d",hap[a]);
#endif
		      
		    for(a=0;a<marker;++a)
		      MsM[side][a][k]=hap[a];

		  }
	      }/*Next tree branching*/
  
		
		  
		
	     
	} /*tree is done*/



      w=wild-pen1*wild+pen2*muta;
      m=muta-pen2*muta+pen1*wild;
	 
	
	  
	  
      /*because of incomplete penetrance, cases are concidered as controls and
	vice versa*/
	  
	  
	
      /* pen1*wild cases are selected*/ 
      for (i=0;i<pen1*wild;++i)
	do   
	  {
	    z1=0;
	    ma[i]=m*ran1(&seed)+1;
	       
	       
	    for(j=0;j<i-1;++j)
	      if (ma[i]==ma[j]) z1=1;
	
	  }
	while (z1==1);
 
      /*pen2*muta controls are selected*/
      for (i=0;i<pen2*muta;++i)

	do   
	  {
	    z1=0;
	    wa[i]=w*ran1(&seed)+1;
	    for(j=0;j<i-1;++j)
	      if (wa[i]==wa[j]) z1=1;
	       
	  }
	while (z1==1);
  
      /*the selected cases are added to the controls*/ 
      for(i=0;i<pen1*wild;++i)
	{
	  w+=1;
	  k=ma[i];

	  for(side=0;side<flanks;++side)
	    for (a=0;a<marker;++a)
	      {
		MsW[side][a][w]=MsM[side][a][k]; /*added as last new wt*/
		for(j=k;j<m;++j)
		  MsM[side][a][j]=MsM[side][a][j+1];/*mt-nr above k are shifted left*/
	      }

	  for(l=i+1;l<=pen1*wild;++l)
	    if(ma[l]>k) --ma[l];	/*transfer-nr shifted as well*/
      
	  m-=1;
	}

      /*the selected controls are added to the cases*/
      for (i=0;i<pen2*muta;++i)
	{
	  m+=1;
	  k=wa[i];

	  for(side=0;side<flanks;++side)
	    for (a=0;a<marker;++a)
	      {
		MsM[side][a][m]=MsW[side][a][k];
		for(j=k;j<w;++j)
		  MsW[side][a][j]=MsW[side][a][j+1];
	      }

	  for(l=i+1;l<=pen2*muta;++l)
	    if(wa[l]>k) --wa[l];
	     
	  w-=1;
	}
	 
	 
      /*#####Statistik#####*/
      /*"""""""""""""""""""*/
	 
      /*Setting all Counters to Zero*/

      for(side=0;side<flanks;++side)
	for(a=0;a<marker;++a)
	  {
	    d[side][a][z]=0;
	    for (i=0;i<Allele[side][a];++i)
	      {
		AnzM[side][a][i]=0;
		AnzW[side][a][i]=0;
	      }
	  }

      maxd[z]=0;

      if(habmar=='y'||habmar=='Y')
	{
	  for(side=0;side<flanks;++side)
	    for(side2=0;side2<flanks;++side2)
	      for(a=0;a<marker;++a)
		for(b=0;b<marker;++b)
		  {
		    dh[side][a][side2][b][z]=0;
		    for(i=0;i<Allele[side][a];++i)
		      for(j=0;j<Allele[side2][b];++j)
			{	
			  AnzHW[side][a][side2][b][i][j]=0;
			  AnzHM[side][a][side2][b][i][j]=0;
			}
		  }
	}

      /*Analysis of single markers*/
      /*""""""""""""""""""""""""""*/

      /*Counting the marker-allels*/
      for(side=0;side<flanks;++side)
	for(a=0;a<marker;++a)
	  for (i=0;i<Allele[side][a];++i)
	    {
	      for (j=1;j<=m;++j) if (MsM[side][a][j]==i) AnzM[side][a][i]+=1;
	      for (j=1;j<=w;++j) if (MsW[side][a][j]==i) AnzW[side][a][i]+=1;
	    }
	 
      /*if marker 0 at flank 0 is monomorphic in the whole sample, the sim is repeated*/ 
      Test1=0;
      for(i=0;i<Allele[0][0];++i)
	if(AnzW[0][0][i]==wild && AnzM[0][0][i]==muta)Test1=1;	    
	   
      if(Test1==1)
	{
	  z=z-1;
	  NIMarker+=1; 
	}
      else
	{
	  af=fopen(outfile,"a");
	  if (hapaus=='y'||hapaus=='Y')
	    {
	      /*Printing the Haplotype to file*/
	      /*Only works for exactly one or two flanks*/

	      for(k=1;k<=m;++k)
		{  
		  fprintf(af,"\n");
		  if (flanks ==2)
		    for(a=0;a<marker;++a)
		      fprintf(af,"%d",MsM[1][a][k]);
		  fprintf(af,"1");
		  for(a=0;a<marker;++a)
		    fprintf(af,"%d",MsM[0][a][k]);
		}
	      for(k=1;k<=w;++k)
		{ 
		  fprintf(af,"\n");
		  if (flanks == 2)
		    for(a=0;a<marker;++a)
		      fprintf(af,"%d",MsW[1][a][k]);
		  fprintf(af,"0");
		  for(a=0;a<marker;++a)
		    fprintf(af,"%d",MsW[0][a][k]);
		}
	    
	    }
	     

	  /*Calculating all d-values of single markers*/
	  for(side=0;side<flanks;++side)
	    for(a=0;a<marker;++a)
	      {
		for(i=0;i<Allele[side][a];++i)
		  {
		    hw=(float)AnzW[side][a][i]/w;
		    hm=(float)AnzM[side][a][i]/m;
		 
		    d[side][a][z]+=(hm-hw)*(hm-hw);

		  }
	     
      
		d[side][a][z]=sqrt(d[side][a][z]);
		 
		if (hapaus=='y'||hapaus=='Y')
		  fprintf(af,"\nd (flank %d, marker %d): %f",side+1,a+1,d[side][a][z]);

		Schnitt[side][a]+=d[side][a][z];

		/*looking for maximal d-value*/
		if(maxmar=='y')
		  if(d[side][a][z]>maxd[z])
		    maxd[z]=d[side][a][z];
		    
		   
	      }

	  /*Statistic for marker-haplotypes*/
	  /*"""""""""""""""""""""""""""""""*/

	  /* Counting Haplotypes*/
	  if(habmar=='y'||habmar=='Y')
	    {
	      if(flanks==2)
		/*counting haplotypes on the left (2)-flank*/
		for(a=0;a<marker;++a)
		  {
		    for(b=a+1;b<marker;++b)
		      {
			for(i=0;i<Allele[1][a];++i)
			  for(j=0;j<Allele[1][b];++j)
			    {
			      for (k=1;k<=m;++k) 
				if (MsM[1][a][k]==i && MsM[1][b][k]==j) 
				  AnzHM[1][a][1][b][i][j]+=1;
			      for (k=1;k<=w;++k) 
				if (MsW[1][a][k]==i && MsW[1][b][k]==j) 
				  AnzHW[1][a][1][b][i][j]+=1;
			    }
		      }
		    /*counting haplotypes with one marker on the right and one marker on the left flank*/
		    for(b=0;b<marker;++b)
		      {
			for(i=0;i<Allele[1][a];++i)
			  for(j=0;j<Allele[0][b];++j)
			    {
			      for (k=1;k<=m;++k) 
				if (MsM[1][a][k]==i && MsM[0][b][k]==j) 
				  AnzHM[1][a][0][b][i][j]+=1;
			      for (k=1;k<=w;++k) 
				if (MsW[1][a][k]==i && MsW[0][b][k]==j) 
				  AnzHW[1][a][0][b][i][j]+=1;
			    }
		      }
	     
		
		  }	     
		 
	      /*counting haplotypes on the right (0)-flank*/
	      for(a=0;a<marker-1;++a)
		for(b=a+1;b<marker;++b)
		  {
		    for(i=0;i<Allele[0][a];++i)
		      for(j=0;j<Allele[0][b];++j)
			{
			  for (k=1;k<=m;++k) 
			    if (MsM[0][a][k]==i && MsM[0][b][k]==j) 
			      AnzHM[0][a][0][b][i][j]+=1;

			  for (k=1;k<=w;++k) 
			    if (MsW[0][a][k]==i && MsW[0][b][k]==j) 
			      AnzHW[0][a][0][b][i][j]+=1;	
			}	     
		  }
	       


	      if (hapaus=='y'||hapaus=='Y')
		/*prints haplotype arround mutation*/
		for(i=0;i<Allele[1][0];++i)
		  for(j=0;j<Allele[0][0];++j)
		    {
		      fprintf(af,"\nHaplotyp %d -0- %d :%d",
			      i,j,AnzHW[1][0][0][0][j][i]);
		      fprintf(af,"\nHaplotyp %d -1- %d :%d",
			      i,j,AnzHM[1][0][0][0][j][i]);
		    }
	 
	    
	      /*Calculating d-values for haplotypes*/
	      if(flanks==2)  
		for(a=0;a<marker;++a)
		  {
		    for(b=a+1;b<marker;++b)
		      {
			for(i=0;i<Allele[1][a];++i)
			  for(j=0;j<Allele[1][b];++j)
			    {
			      hw=(float)AnzHW[1][a][1][b][i][j]/w;
			      hm=(float)AnzHM[1][a][1][b][i][j]/m;
	      
			      dh[1][a][1][b][z]+=(hm-hw)*(hm-hw);
			    }
		       

			dh[1][a][1][b][z]=sqrt(dh[1][a][1][b][z]);
			SchnittH[1][a][1][b]+=dh[1][a][1][b][z];
		      }	
	 
		    for(b=0;b<marker;++b)
		      {
			for(i=0;i<Allele[1][a];++i)
			  for(j=0;j<Allele[0][b];++j)
			    {
			      hw=(float)AnzHW[1][a][0][b][i][j]/w;
			      hm=(float)AnzHM[1][a][0][b][i][j]/m;
	      
			      dh[1][a][0][b][z]+=(hm-hw)*(hm-hw);
			    }
		       

			dh[1][a][0][b][z]=sqrt(dh[1][a][0][b][z]);
			SchnittH[1][a][0][b]+=dh[1][a][0][b][z];
		      }
		  }

	      for(a=0;a<marker-1;++a)
		for(b=a+1;b<marker;++b)
		  {
		    for(i=0;i<Allele[0][a];++i)
		      for(j=0;j<Allele[0][b];++j)
			{
			  hw=(float)AnzHW[0][a][0][b][i][j]/w;
			  hm=(float)AnzHM[0][a][0][b][i][j]/m;
	      
			  dh[0][a][0][b][z]+=(hm-hw)*(hm-hw);
			}
		       
		    dh[0][a][0][b][z]=sqrt(dh[0][a][0][b][z]);
		    SchnittH[0][a][0][b]+=dh[0][a][0][b][z];
		  }
	    }
	       
	  if(z==0 && silent ==0)
	    {
	      fprintf(stderr,"\ntime for first loop: %d s",time(NULL)-AZeit);
	      fprintf(stderr,"\nexpected time for simulation: %.3f h\n\n",
		      (float)Wiederholungen*(time(NULL)-AZeit)/3600); 
	    }

	
	
	
	
	  fclose(af);
	}
      if (silent ==0)
	fprintf(stderr,"loop: %d\n",z+1);

    }/*end of repeat*/ 

    
      






  /*		Generating the Distribution		*/
  /*		"""""""""""""""""""""""""""    		*/
      
  
  af=fopen(outfile,"a");

  for(side=0;side<flanks;++side)
    for(a=0;a<marker;++a)
      {
	
	for (i=0;i<steps*sqrt(2);++i)
	  {
	    Co[i]=0;
	    for (j=0;j<Wiederholungen;++j)
	      if (d[side][a][j]<(float)(i+1)/steps && d[side][a][j]>=(float)i/steps) 
		Co[i]+=1;
	  }
	
	
	/*Generating stat. Varianz and std. deviation*/

	Var=0;
	Schnitt[side][a]=Schnitt[side][a]/Wiederholungen;
	for (j=0;j<Wiederholungen;++j)
	  Var+=(Schnitt[side][a]-d[side][a][j])*(Schnitt[side][a]-d[side][a][j]);
	Var=Var/(Wiederholungen-1);
	sigma=sqrt(Var);
	
	
	 
	    
	fprintf (af,"\n\nflank %d:  marker %d:",side+1,a+1);
	fprintf (af,"\n---------");
	fprintf (af,"\naverage: %.3f",Schnitt[side][a]);
	fprintf (af,"\nvariance: %.3f\tstandard deviation: %.3f",Var,sigma);


	fprintf (af,"\ndistribution:");
	Sum=0;
	for (i=0;i<steps*sqrt(2);++i)
	  {
	    if (Co[i]>0) fprintf (af,"\n%.3f-%.3f:\t%d",
				  (float)i/steps,(float)(i+1)/steps,Co[i]);
	    Sum+=Co[i];
	  }
	if(side==0 && a==0)
	  fprintf (af,"\nnumber of un-informative sims:\t%d",NIMarker);
      }

#ifdef DEBUGs
  fprintf (af,"\ndistribution of the last repeat:");
  for(side=0;side<flanks;++side)
    for(a=0;a<marker;++a)
      for(i=0;i<Allele[side][a];++i)
	fprintf (af,"\nside %d marker %d allele %d: M:%d W:%d",
		 side,a,i,AnzM[side][a][i],AnzW[side][a][i]); 
#endif DEBUGs

  /*generating the distribution of the maximal marker*/
  if(maxmar=='y')
    { 
      MaAv=0;Var=0;sigma=0;Sum=0;
	  
      for (j=0;j<Wiederholungen;++j)
	MaAv+=maxd[j];
	  
      MaAv=MaAv/Wiederholungen;

      for (j=0;j<Wiederholungen;++j)
	Var+=(MaAv-maxd[j])*(MaAv-maxd[j]);

      Var=Var/(Wiederholungen-1);
      sigma=sqrt(Var);
	

      for (i=0;i<steps*sqrt(2);++i)
	{
	  Co[i]=0;
	  for (j=0;j<Wiederholungen;++j)
	    if (maxd[j]<(float)(i+1)/steps && maxd[j]>=(float)i/steps) Co[i]+=1;
	}

      fprintf (af,"\n\ndistribution of the maximal marker:");
      fprintf (af,"\naverage: %.3f",MaAv);
      fprintf (af,"\nvariance: %.3f\tstandard deviation: %.3f",Var,sigma);

      for (i=0;i<steps*sqrt(2);++i)
	{
	  if (Co[i]>0) fprintf (af,"\n%.3f-%.3f:\t%d",
				(float)i/steps,(float)(i+1)/steps,Co[i]);
	  Sum+=Co[i];
	}
	
      
	  
      for(side=0;side<flanks;++side)
	for(a=0;a<marker;++a)
	  maz[side][a]=0;

      for(j=0;j<Wiederholungen;++j)
	for(side=0;side<flanks;++side) 
	  for(a=0;a<marker;++a)
	    if (maxd[j]==d[side][a][j]) maz[side][a]+=1;

      fprintf (af,"\nnumber of times marker carrying maximal info:");
      for(side=0;side<flanks;++side)
	for(a=0;a<marker;++a)
	  fprintf (af,"\nside %d marker %d :\t%d",side+1,a+1,maz[side][a]);
    }
  /*Distribution of haplotypes*/
      
  if(habmar=='y'||habmar=='Y')
    {
      if(flanks==2)
	{
	  for(a=0;a<marker-1;++a)	      
	      
	    for(b=a+1;b<marker;++b)
	      {
		for (i=0;i<steps*sqrt(2);++i)
		  {
		    Co[i]=0;
		    for (j=0;j<Wiederholungen;++j)
		      if (dh[1][a][1][b][j]<(float)(i+1)/steps && dh[1][a][1][b][j]>=(float)i/steps) 
			Co[i]+=1;
		  }
	
		Var=0;
		SchnittH[1][a][1][b]=SchnittH[1][a][1][b]/Wiederholungen;
		for (j=0;j<Wiederholungen;++j)
		  Var+=(SchnittH[1][a][1][b]-dh[1][a][1][b][j])*
		    (SchnittH[1][a][1][b]-dh[1][a][1][b][j]);
		Var=Var/(Wiederholungen-1);
		sigma=sqrt(Var);
	
		fprintf (af,"\n\nhaplotype on flank 2 markers %d, %d:",a+1,b+1);
		fprintf (af,"\n---------");
		fprintf (af,"\naverage: %.3f",SchnittH[1][a][1][b]);
		fprintf (af,"\nvariance: %.3f\tstandard deviation: %.3f",Var,sigma);
	
		fprintf (af,"\ndistribution:");
		Sum=0;
		for (i=0;i<steps*sqrt(2);++i)
		  {
		    if (Co[i]>0) fprintf (af,"\n%.3f-%.3f:\t%d",
					  (float)i/steps,(float)(i+1)/steps,Co[i]);
		    Sum+=Co[i];
		  }
		   
		   
		fprintf(af,"\n\n");
	
	      }
	  for(a=0;a<marker;++a)
	    for(b=0;b<marker;++b)
	      {
		for (i=0;i<steps*sqrt(2);++i)
		  {
		    Co[i]=0;
		    for (j=0;j<Wiederholungen;++j)
		      if (dh[1][a][0][b][j]<(float)(i+1)/steps && dh[1][a][0][b][j]>=(float)i/steps) 
			Co[i]+=1;
		  }
	
	
		Var=0;
		SchnittH[1][a][0][b]=SchnittH[1][a][0][b]/Wiederholungen;
		for (j=0;j<Wiederholungen;++j)
		  Var+=(SchnittH[1][a][0][b]-dh[1][a][0][b][j])*(SchnittH[1][a][0][b]-dh[1][a][0][b][j]);
		Var=Var/(Wiederholungen-1);
		sigma=sqrt(Var);
	
	
		fprintf (af,"\n\nhaplotype on flank 2  marker %d;  flank 1 marker %d:",a+1,b+1);
		fprintf (af,"\n---------");
		fprintf (af,"\naverage: %.3f",SchnittH[1][a][0][b]);
		fprintf (af,"\nvariance: %.3f\tstandard deviation: %.3f",Var,sigma);
	
		fprintf (af,"\ndistribution:");
		Sum=0;
		for (i=0;i<steps*sqrt(2);++i)
		  {
		    if (Co[i]>0) fprintf (af,"\n%.3f-%.3f:\t%d",
					  (float)i/steps,(float)(i+1)/steps,Co[i]);
		    Sum+=Co[i];
		  }
		   
		    
		fprintf(af,"\n\n");
	
	      }
	}


      for(a=0;a<marker-1;++a)
	for(b=a+1;b<marker;++b)
	  {
	    for (i=0;i<steps*sqrt(2);++i)
	      {
		Co[i]=0;
		for (j=0;j<Wiederholungen;++j)
		  if (dh[0][a][0][b][j]<(float)(i+1)/steps && dh[0][a][0][b][j]>=(float)i/steps) 
		    Co[i]+=1;
	      }
      
	
	    Var=0;
	    SchnittH[0][a][0][b]=SchnittH[0][a][0][b]/Wiederholungen;
	    for (j=0;j<Wiederholungen;++j)
	      Var+=(SchnittH[0][a][0][b]-dh[0][a][0][b][j])*(SchnittH[0][a][0][b]-dh[0][a][0][b][j]);
	    Var=Var/(Wiederholungen-1);
	    sigma=sqrt(Var);
	
	
	    fprintf (af,"\n\nhaplotype on flank 1  markers %d, %d:",a+1,b+1);
	    fprintf (af,"\n---------");
	    fprintf (af,"\naverage: %.3f",SchnittH[0][a][0][b]);
	    fprintf (af,"\nvariance: %.3f\tstandard deviation: %.3f",Var,sigma);
	
	    fprintf (af,"\ndistribution:");
	    Sum=0;
	    for (i=0;i<steps*sqrt(2);++i)
	      {
		if (Co[i]>0) fprintf (af,"\n%.3f-%.3f:\t%d",
				      (float)i/steps,(float)(i+1)/steps,Co[i]);
		Sum+=Co[i];
	      }
		
	
	    fprintf(af,"\n\n");
	
	  }
    }
  fprintf(af,"\n");
  fclose(af);


  
  
}/*end of main*/






/*Bestimmung des naechsten Ereignisses*/


double p1(int , float);
double p2(int ,float);
double p3(int ,double);
int Ereignis (int w, int m, double mue,float t)
{
 
  double Nenner;
 
  double pm,pw,pc;
  
  pw=p1(w,t);
  pm=p2(m,t);
  pc=p3(m,mue);
      
  Nenner=pm+pw+pc;
  
  if (test(pw/Nenner)==0) return (1);
  else if (test(pm/(pm+pc))==0) return (2);
  
  else return (3);
    
 

}



/* rate for coalescent in Control population*/ 

double p1(int wild, float t)
{
  if (wild>1)return ((wild)*(wild-1)/(2*pop(t)));
  else return (0);
}


/* rate for coalescent in Case population*/ 
double p2(int muta, float t)
{
  
  if (muta>1) return ((muta)*(muta-1)/(2*preva*pop(t)));
  else return(0);

}

/* rate for Mutation*/ 
double p3(int muta, double mu)
{
  if (muta>0) return (mu*muta);
  else return(0);

}



/*Random Value, between 1 an 0 NRC, S.280*/
#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define EPS 1.2e-7
#define RNMX (1.0-EPS)

float ran1(long *idum)
{
	int j;
	long k;
	static long iy=0;
	static long iv[NTAB];
	float temp;

	if (*idum <= 0 || !iy) {
		if (-(*idum) < 1) *idum=1;
		else *idum = -(*idum);
		for (j=NTAB+7;j>=0;j--) {
			k=(*idum)/IQ;
			*idum=IA*(*idum-k*IQ)-IR*k;
			if (*idum < 0) *idum += IM;
			if (j < NTAB) iv[j] = *idum;
		}
		iy=iv[0];
	}
	k=(*idum)/IQ;
	*idum=IA*(*idum-k*IQ)-IR*k;
	if (*idum < 0) *idum += IM;
	j=iy/NDIV;
	iy=iv[j];
	iv[j] = *idum;
	if ((temp=AM*iy) > RNMX) return RNMX;
	else return temp;
}
#undef IA
#undef IM
#undef AM
#undef IQ
#undef IR
#undef NTAB
#undef NDIV
#undef EPS
#undef RNMX




/*Berechnung der Populationsgroesse zu einem Zeitpunkt t*/

 double pop(float t)
{
  if (rho==1.0) return (N0);
   else
    {
      if (t>=tau) return (N0/rho);
      else return (N0*exp(-log(rho)*t/tau));
    }
}
/*Berechnung der kumulativen Dichte zu einem Zeitpunkt t*/

double Lambda(float t)
{
  if (rho==1.0) return (t);
  if (t==0.0) return (0.0);
  else
    {
      if (t<=tau) return( (tau/log(rho))*(exp(t*log(rho)/tau)-1.0));
      else return ( (tau*(rho-1.0)/log(rho))+rho*(t-tau)  );
    }
}




double zeroin(double , double , double ,double ,double );
void zbrac(double, double, double);
double Lambda(float);
float OG;
float UG;

double WaTime(int w,int m,double mu,float t)
      
  
{	
  double a,b,c;
  double dum,lrho;
  double F,G;
  double V,K,ZW;
  double LSG;
  FILE *fp;
 
  a=0.0;b=0.0;c=0.0;
  F=(float)w*(w-1)/2+(float)m*(m-1)/(2*preva);

  V=Lambda(t);

  do
    dum=ran1(&seed);
  while (dum == 0.0 || dum == 1.0);

  dum=log(dum);
 	

	
  if (rho==1.0) LSG= ((-1.0)*dum/(m*mu+F/N0));
  else
    {
        lrho=log(rho);
	
	G=(m*mu*t+F*V/N0)-(mu*m*tau+F/N0*(tau*(rho-1)/lrho));           /*dum<G <=> tneu>tau*/
      if (dum<=G) 
	LSG=(-1.0)*((N0*dum)+F*(tau*(rho-1)/lrho+rho*(t-tau)-V))/(rho*F+m*mu*N0);
      else

	{
	  K=exp(t*lrho/tau);
	  
	  
	  
	  ZW=V/K*lrho/tau;
	  c=(N0*lrho*dum)/(F*tau)-1.0;
	  c=c/K -ZW;
	  
	   
	  
	  b=(m*mu*N0*lrho)/(F*tau*K);
	  a=lrho/tau;
	  
	  zbrac(a,b,c);

	  LSG = zeroin(0,OG,a,b,c);
	}
	  if (LSG<=0)  
	    {
	      
	      fp=fopen("Error","a");	
	      fprintf(fp,"\nw:%d",w);
	      fprintf(fp,"\nm:%d",m);
	      fprintf(fp,"\nt:%f",t);	
	      fprintf(fp,"\ndum:%f",dum);
	      fprintf(fp,"\nN0:%e",N0);
	      fprintf(fp,"\nlrho:%f",lrho);
	      fprintf(fp,"\nF:%f",F);
	      fprintf(fp,"\ntau:%f",tau);
	      fprintf(fp,"\nV:%e",V);
	      fprintf(fp,"\nV(tau):%e",tau*(rho-1)/lrho);
	      fprintf(fp,"\nK:%f\n",K);
	      fprintf(fp,"\nZW:%f\n",ZW);
	      fprintf(fp,"\na :%f",a); 
	      fprintf(fp,"\nb:%f",b); 
	      fprintf(fp,"\nc:%f",c); 
	      fprintf(fp,"\nUG:%f",UG); 
	      fprintf(fp,"\nOG:%f\n",OG); 

	      fprintf(fp,"\nLSG:%f\n",LSG); 
	      fclose(fp); 
	      return(0);
	     		
	    }	
	  
    }
   
     /*Aenderung hier richtig??*/    
  return (LSG);	
    
  }

 /*Approximiert Nullstelle fuer exp(xt)+yt+z*/
#define TOLERANCE 1.0e-10

double zeroin(double ax, double bx, double x,double y,double z)
   {
     double a,b,c;				/* Abscissae	*/
     double fa;				/* f(a)			*/
     double fb;				/* f(b)			*/ 
     double fc;				/* f(c)			*/

     a = ax;  b = bx;  fa = exp(a*x)+a*y+z;  fb = exp(b*x)+b*y +z;
     c = a;   fc = fa;

     for(;;)		/* Main iteration loop	*/
       {
	 double prev_step = b-a;	/* Distance from the last but one*/
							/* to the last approximation	*/
	 double p;      			/* Interpolation step is calcu- */
	 double q;      			/* lated in the form p/q; divi- */
  							/* sion operations is delayed   */
 							/* until the last moment		*/
	 double new_step;      	/* Step at this iteration       */

    if( fabs(fc) < fabs(fb) )
    {                         		/* Swap data for b to be the 	*/
	a = b;  b = c;  c = a;          /* best approximation			*/
	fa=fb;  fb=fc;  fc=fa;
    }
    new_step = (c-b)/2;

    if( fabs(new_step) <= TOLERANCE || fb == (double)0 )
      return b;				/* Acceptable approx. is found	*/

    			/* Decide if the interpolation can be tried	*/
    if( fabs(prev_step) >= TOLERANCE	/* If prev_step was large enough*/
	&& fabs(fa) > fabs(fb) )	/* and was in true direction,	*/
    {					/* Interpolatiom may be tried	*/
	register double t1,cb,t2;
	cb = c-b;
	if( a==c )			/* If we have only two distinct	*/
	{				/* points linear interpolation 	*/
	  t1 = fb/fa;			/* can only be applied		*/
	  p = cb*t1;
	  q = 1.0 - t1;
 	}
	else				/* Quadric inverse interpolation*/
	{
	  q = fa/fc;  t1 = fb/fc;  t2 = fb/fa;
	  p = t2 * ( cb*q*(q-t1) - (b-a)*(t1-1.0) );
	  q = (q-1.0) * (t1-1.0) * (t2-1.0);
	}
	if( p>(double)0 )	/* p was calculated with the op-*/
	  q = -q;			/* posite sign; make p positive	*/
	else				/* and assign possible minus to	*/
	  p = -p;			/* q				*/

	if( p < (0.75*cb*q-fabs(TOLERANCE*q)/2)	/* If b+p/q falls in [b,c]*/
	    && p < fabs(prev_step*q/2) )	/* and isn't too large	*/
	  new_step = p/q;					/* it is accepted	*/
    }

    if( fabs(new_step) < TOLERANCE )	/* Adjust the step to be not less*/
      if( new_step > (double)0 )	/* than tolerance		*/
	new_step = TOLERANCE;
      else
	new_step = -TOLERANCE;

    a = b;  fa = fb;			/* Save the previous approx.	*/
    b += new_step;  fb = exp(b*x)+b*y+z;	/* Do step to a new approxim.	*/
    if( (fb > 0 && fc > 0) || (fb < 0 && fc < 0) )
    {                 			/* Adjust c for it to have a sign*/
      c = a;  fc = fa;                  /* opposite to that of b	*/
    }
  }

}

#undef TOLERANCE




#define FACTOR 1.6

void zbrac(double a, double b, double c)
{
	
	int j;
	float f1,f2;
	float x1,x2;

        x1=(double)0;
	x2=(double)1;
	f1=exp(a*x1)+b*x1+c;
	f2=exp(a*x2)+b*x2+c;
	while(f1*f2 >= 0.0)
	 {
		
		if (fabs(f1) < fabs(f2))
			{
                         x1 += FACTOR*(x1-x2);
			f1=exp(a*x1)+b*x1+c;
			}	
		else
			{			
			x2 += FACTOR*(x2-x1);
			f2=exp(a*x2)+b*x2+c;
			}
	}
	if(x1<x2)
		{
		UG=x1;
		OG=x2;
		}
	else
		{
		UG=x2;
		OG=x1;
		}	
}
#undef FACTOR

short test(double war)
{
if (war>=1e-6)
  if (ran1(&seed)<=war) return (0);
  else return(1); 
else
  if (ran1(&seed)<=1e+5)
    {
      war=war*1e+5;
      if(ran1(&seed)<=war) return (0);
      else return(1);
    }
  else return(1);
}



void recomb(char side)
{
  int i,j,k;
  char coflag[MARKER+1]; /*Flags if recomb between i-1 und i*/
 


 /*Checking for recombinations*/
  for(i=0;i<marker;++i)
    {

      if (ran1(&seed)<=rec[side][i]) 
	{
	  coflag[i]=1;
#ifdef DEBUG
	  printf("\nRekomb: %d",i);
#endif
	}
      else coflag[i]=0;	
    }
  coflag[marker]=1;	/*for an uneven number of recombinations*/


  /*Switching the area between two recombinations*/
  for(i=0;i<marker;++i)
    if (coflag[i]==1)
      {
      j=i+1;
      
      while (coflag[j]==0)
	  ++j;
      
      /*Rekombination zwischen i-1 und i und j-1 und j*/
      drawhap(side,i,j);
      
      for(k=i;k<j;++k)
	hap[k]=newhap[k];


      i=j+1;
      }

return;
}

void drawhap(char side,int markerA,int markerE)
     /*Generates a new subhaplotype between markerA(incl) and markerE(excl)*/
     /*And writes it into newhap[]*/

{
  char ghap[MARKER+1];	/*counter for all haplotypes*/
  char nh[MARKER+1]; 	/*counter for all haplotypes between markerA and markerE*/
  int i;
  double ZS; /*The probability of a single haplotype*/
  float zz;
  double CummuProb=0.0;


#ifdef DEBUG
  printf("\nin drawhap:%d-%d\n",markerA,markerE);
#endif

  zz=ran1(&seed);/*Generating a random Number between 0 and 1*/

  for(i=0;i<=marker;++i)
    {
    nh[i]=0;
    }

  while (nh[markerE]==0)
  {
    for(i=0;i<=marker;++i)
      ghap[i]=0;
    
    for(i=markerA;i<markerE;++i)
      ghap[i]=nh[i];
    
    ZS=0;
    
    /*the prb of all haplotypes with the sub hap nh is added*/ 
    while (ghap[marker]==0)
      {
	ZS+=prob(side,ghap);	
	
	if (markerA>0)
	{
	  ++ghap[0];
	  for(i=0;i<(markerA-1);++i)
	    if (ghap[i]==Allele[side][i])
	      {
		ghap[i]=0;
		++ghap[i+1];
	      }
	  if(ghap[markerA-1]==Allele[side][markerA-1])
	    {
	      ghap[markerA-1]=0;
	      ghap[markerE]+=1;
	    }
	}
	else ++ghap[markerE];

	for(i=markerE;i<marker;++i)
	  if (ghap[i]==Allele[side][i])
	    {
	      ghap[i]=0;
	      ghap[i+1]+=1;
	    }				
    } 

    CummuProb+=ZS;

#ifdef DEBUG
	printf("\nCummuProb:%f",CummuProb);
#endif

    if (zz<CummuProb)
      {

#ifdef DEBUG
	printf("\nTreffer: %d-%d:",markerA,markerE);
	for(i=markerA;i<markerE;++i)
	  printf("\t%d",nh[i] );
#endif
      break;
      }
    
    /*new nh*/
    nh[markerA]+=1;

    for(i=markerA;i<markerE;++i)
      if (nh[i]==Allele[side][i])
	{
	  nh[i]=0;
	  nh[i+1]+=1;
	}
    
    
  }


  for(i=markerA;i<markerE;++i)
    newhap[i]=nh[i];

  return;

}

/*Generates the frequency of haplotype ha*/ 

double prob(char side,char ha[MARKER])
{
  int i;
  int HZ=0;

  

  for(i=0;i<marker;++i)
    HZ+=pow(MAXALLEL,i)*ha[i];


  return(freq[side][HZ]);
}



/*generates an equal LE-distribution of marker-haplotypes*/
void eqdist(void)
{
  int i;
  char side;
  char ghap[(MARKER+1)];
  int HZ; /*Haplotype counter*/
  /*HZ represents the haplotype 1*state[0]+MAXALLEL*state[1]+MAXALLEL^2*state[2]....*/

  for(side=0;side<flanks;++side)
    {  
      for(i=0;i<=marker;++i)
	ghap[i]=0;


      while (ghap[marker]==0)
	{
	  HZ=0;
	  for(i=0;i<marker;++i)
	    HZ+=pow(MAXALLEL,i)*ghap[i];

	  freq[side][HZ]=1.0;
      
	  for(i=0;i<marker;++i)
	    freq[side][HZ]=freq[side][HZ]/Allele[side][i];
      
	       
#ifdef DEBUG
	 
	  for(i=0;i<=marker;++i)
	    printf("side %d %d  ",side,ghap[i]);

	  printf("  %f\n",freq[side][HZ]);
#endif 
      

	  ghap[0]+=1;


	  for(i=0;i<marker;++i)
	    if (ghap[i]==Allele[side][i])
	      {
		ghap[i]=0;
		ghap[(i+1)]+=1;
	      }
	
      
	}
    }
#ifdef DEBUG
  printf("\neqdist fertig\n");
#endif
  return;
 
}


void popuinput(char mode)
     /*changes values of N0,rho,tau*/
{
  char line[100];

  float Nin,tauin,prin,Norg;
 
  switch (mode)
    {
    case 'r':
      N0=6e12;
      tau=500.0;
      rho=6e8;
      preva=1.0;
      break;
    case 'o':
      N0=6e12;
      tau=2500.0;
      rho=6e8;
      preva=1.0;
      break;
    case 'c':
      N0=1e4;
      tau=10.0;
      rho=1.0;
      preva=1.0;
      break;
    default:
      do{
	fprintf(stdout,"\ninput present population size:\n");
	fgets(line,sizeof(line),stdin);
	sscanf(line,"%e",&Nin);
#ifdef DEBUG
	fprintf(stderr,"\n%e",Nin);
#endif 
	if (Nin<=0.0) fprintf(stderr,"\nnot a positive number\n"); 
      }while(Nin<=0.0);

      do
	{
	  fprintf(stdout,"\ninput starting population size:\n");
	  fgets(line,sizeof(line),stdin);
	  sscanf(line,"%e",&Norg);
	  if (Norg<=0.0) fprintf(stderr,"\nnot a positive number\n");
	}
      while (Norg<=0.0);

      if(Norg!=Nin)
	do
	{
	  fprintf(stdout,"\ninput time to the start of expansional phase:\n");
	  fgets(line,sizeof(line),stdin);
	  sscanf(line,"%f",&tauin);
	  if (tauin<=0.0) fprintf(stderr,"\nnot a positive number\n");
	}
	while (tauin<=0.0);

      do
	{
	  fprintf(stdout,"\ninput disease frequency f\n");
	  fgets(line,sizeof(line),stdin);
	  sscanf(line,"%f",&prin);
	  if (prin<=0.0) fprintf(stderr,"\nnot a positive number\n");
	}
      while (prin<=0.0);
#ifdef DEBUG
      fprintf(stderr,"\n%f\t%f\t%f\t%f\n",Nin,Norg,tauin,prin);
#endif 
      tau=tauin;
      N0=Nin;
      rho=N0/Norg;
      preva=prin;
      break;
    }
  return;
}

int sasizeinput(char type)
{
  int erg;
  char line[100];
  short ok;
  do
    {
      ok=0;
      switch (type)
	{
	case 'w':
	  fprintf(stdout,"\ninput control sample size\n");
	  break;
	case 'm':
	  fprintf(stdout,"\ninput case sample size\n");
	  break;
	default:
	  break;
	}
      fgets(line,sizeof(line),stdin);
      sscanf(line,"%d",&erg);

      if (erg<=0) 
	{
	  fprintf(stderr,"\nnot a positive number\n");
	  ok=1;
	}
      if (erg>MAXSAMPLE) 
	{
	  fprintf(stderr,"\nnumber too large\n");
	  ok=1;
	}
    }
  while (ok==1);
  return(erg);
}

float peninput(char type)

{
  float erg;
  char line[100];
  short ok;
  do
    {
      ok=0;
      switch (type)
	{
	case 'a':
	  fprintf(stdout,"\ninput frequency of phenotypical controls that have the ancestry of a case\n");
	  break;
	case 'o':
	  fprintf(stdout,"\ninput frequency of phenotypical cases that have the ancestry of a control\n");
	  break;
	default:
	  break;
	}
      fgets(line,sizeof(line),stdin);
      sscanf(line,"%f",&erg);

      if (erg<0.0) 
	{
	  fprintf(stderr,"\nthis value cannot be negative\n");
	  ok=1;
	}
      if (erg>1.0) 
	{
	  fprintf(stderr,"\nthis value cannot be >1\n");
	  ok=1;
	}
    }
  while (ok==1);
  return(erg);
}

void usage(char * pn)
{
  fprintf(stderr,"\nusage is %s [option 1] [option2] etc.",pn);
  fprintf(stderr,"\noptions:");
  fprintf(stderr,"\n-a\t\tgenerates LD with all two-marker haplotypes");
  fprintf(stderr,"\n-b\t\tin each loop the best marker is stored. they are jointly analyzed");
  fprintf(stderr,"\n-c\t\tallows the input of a complex phenotype-genotype interaction");
  fprintf(stderr,"\n-d\t\thaplotype data is preserved in file"); 
  fprintf(stderr,"\n-e<distance>\tsets genetic distance between markers in even distribution");
  fprintf(stderr,"\n-f<number>\tdefines wether 1 or both(2) flanks of disease gene \n\t\tare simulated");
  fprintf(stderr,"\n-h/-H\t\tshows this list");
  fprintf(stderr,"\n-l<number>\tsets the number of times the simulation is looped");
  fprintf(stderr,"\n-m<rate>\tsets the mutation rate per generation of the disease gene");
  fprintf(stderr,"\n-n<number>\tsets the number of markers that are simulated on each flank");
  fprintf(stderr,"\n-o<name>\tset output file name");
  fprintf(stderr,"\n-p<model>\tselects population model");
  fprintf(stderr,"\n-r\t\tallows for individual input of recombination frequencies");
  fprintf(stderr,"\n-s\t\tallows for the input of a different sample size\n\t\tthan the default"); 
  
  

  fprintf(stderr,"\n-L\t\tenables input of different haplotype frequencies");
  fprintf(stderr,"\n-M<probability>\tsets mutation probability of each marker");
  fprintf(stderr,"\n-S\t\ttiggers silent mode");
  fprintf(stderr,"\n\n");
  exit(8); 
}

void haplofrequinput(void)
{
  char ha[MARKER+1];
  float fs[MARKER];
  char line[100];
  int i,j,side;
  float erg,sum;
  int HZ;
  short test;

  do
    {
      test=0;
      sum=0.0;
      for(side=0;side<flanks;++side)
	{
	  for(i=0;i<=marker;++i)
	    ha[i]=0;
	  while (ha[marker]==0)
	    {
	      if (sum<1.0)
		{
		  fprintf(stdout,"\ninput frequency of flank %d allele ",side+1);
		  for(i=0;i<marker;++i)
		    fprintf(stdout,"%d",ha[i]);

		  fprintf(stdout,"  ");
		  fgets(line,sizeof(line),stdin);
		  sscanf(line,"%f",&erg);
		  sum+=erg;
		  if(sum>1.0)
		    {
		      fprintf(stderr,"\ninput error: total frequencies >1\n");
		      test=1;
		    }
		}
	 
	      else erg=0.0;
	      HZ=0;
	      for(i=0;i<marker;++i)
		HZ+=pow(MAXALLEL,i)*ha[i];

	      freq[side][HZ]=erg;

	      ha[0]++;
	      for(i=0;i<marker;++i)
		if (ha[i]==Allele[side][i])
		  {
		    ha[i]=0;
		    ++ha[i+1];
		  }
	    }
	}
      if (sum<1.0)
	{
		      fprintf(stderr,"\ninput error: total frequencies <1\n");
		      test=1;
	}
    }
  while(test==1);
  return;
}

void recinput(void)
{
  float erg;
  char line[100];
  short ok;
  int side,i;

  for (side=0;side<flanks;++side)
    for (i=0;i<marker;++i)
      {       
	do
	  {
	    if (i>0)
	      fprintf(stdout,"\ninput recombination probalility on flank %d between marker %d and marker %d\n",side+1,i,i+1);
	    else
	      fprintf(stdout,"\ninput recombination probalility on flank %d between disease gene and marker %d\n",side+1,i+1);
	    fgets(line,sizeof(line),stdin);
	    sscanf(line,"%f",&erg);
	    if (erg<0.0 || erg>=1.0)
	      fprintf(stderr,"input error: recombination probability <0 or >=1");
	  }
	while (erg<0.0 || erg>=1.0);
	rec[side][i]=erg;     
      }
  return;
}


double mutinput(short side,short mar)
{
  char line[100];
  double erg;

  
  do
    
    {
      fprintf(stdout,"\ninput mutation probability of marker %d on side %d\n",mar+1,side+1);
      fgets(line,sizeof(line),stdin);
      sscanf(line,"%lf",&erg);
	    
      if (erg<0.0 || erg>1.0)
	fprintf(stderr,"\ninput error: mutation probability <0 or >1"); 
    }
  while (erg<0.0 || erg>1.0); 

  return (erg);
}
