C **********************************************************************
C ************************ SIBMED.F VERSION 1.0 ************************
C ********************** WRITTEN BY J.A. DOUGLAS ***********************
C *************************  APRIL 18, 2000 ****************************
C **********************************************************************
C
C ORIGINAL:       1/26/99
C LAST MODIFIED:  4/18/00
C
C PURPOSE:  USING ONLY SIB-PAIR GENOTYPE DATA AND A DENSE MARKER MAP,
C DETERMINE WHETHER THE MARKER GENOTYPES FOR EACH PAIR LIKELY REPRESENT
C GENOTYPING ERRORS OR MUTATIONS.
C  
C SPECIFICALLY,
C
C (1) FOR EACH SIB PAIR-MARKER COMBINATION, CALCULATE THE POSTERIOR
C     PROBABILITY THAT AT LEAST ONE SIBLING'S GENOTYPE IS INCORRECT, 
C     CONDITIONAL ON ALL MARKER DATA AND ERROR RATE E AND ASSUMING KNOWN
C     ALLELE FREQUENCIES AND MARKER MAP.
C    
C (2) USING THE MARKER MAP AND ALLELE FREQUENCIES IN (1), SIMULATE 
C     GENOTYPE DATA FOR 1,000,000 SIBLING PAIRS UNDER THE NULL 
C     HYPOTHESIS OF NO GENOTYPING ERROR OR MUTATION AT MARKER K,
C     INTRODUCING ERROR AT RATE E FOR ALL OTHER MARKERS. CALCULATE THE
C     POSTERIOR ERROR PROBABILITY FOR EACH OF THESE.  USING THE
C     PRESCRIBED SIZE OF THE OF THE TEST (FALSE POSITIVE RATE),
C     DETERMINE THE CORRESPONDING POSTERIOR ERROR PROBABILITY CUTOFF
C     CONSTANT.  REPEAT THIS PROCESS FOR EACH MARKER K.
C
C (3) REPORT SIB-PAIR MARKER COMBINATIONS IN (1) WHICH EXCEED THE
C     THE CUTOFF CONSTANTS DETERMINED IN (2).  ALTERNATIVELY, REPORT
C     SIB-PAIR MARKER COMBINATIONS WITH EXCEEDINGLY LARGE POSTERIOR
C     ERROR PROBABILITIES.
C
C REFERENCE: 
C
C     DOUGLAS JA, BOEHNKE M, LANGE K (2000) A MULTIPOINT METHOD FOR
C     DETECTING GENOTYPING ERRORS AND MUTATIONS IN SIBLING-PAIR 
C     LINKAGE DATA.  AM J HUM GENET 66: 1287-1297
C
C AS PART OF THE PROCESS, THE PROGRAM READS MENDEL LOCUS AND PEDIGREE
C FILES AND CONVERTS BAND LENGTH GENOTYPES TO ALLELE NUMBER GENOTYPES
C (1 TO NALL).  
C
C ASSUMPTIONS:  
C 1. AUTOSOMAL, CODOMINANT LOCI.
C 2. NO MORE THAN 40 ALLELES PER MARKER.
C 3. NO MORE THAN 500 MARKERS.
C 4. CHROMOSOMES NEVER OF LENGTH GREATER THAN 50 MORGANS.
C 5. IN EACH GENOTYPE EITHER BOTH ALLELES OR NEITHER ALLELE SPECIFIED.
C 6. KNOWN ERROR RATE E, 0<=E<=1.
C
C KEY TO RELATIVE TYPES:
C 1=FULL SIBS, 2=HALF SIBS.
C
C INITIALIZATIONS.
C
      PARAMETER(MAXALL=40,MAXFAM=2000,MAXLOC=500,MAXPEO=30,MAXREL=2
     1,MAXSIB=10000,MXPTOT=8000,NFRQPR=3,NHAP=2,NSIM=1000000) 
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER*8 ALLNAM(MAXLOC,MAXALL),ID(MXPTOT),LOCNAM(MAXLOC)
     1,P1ID(MXPTOT),P2ID(MXPTOT),PEDID(MXPTOT),SEX(MXPTOT)
     2,TRMODE(MAXLOC),TWIN(MXPTOT),XXSIGN,XYSIGN
      CHARACTER*17 CGENO(MAXPEO,MAXLOC)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),ALPHA(MAXLOC,0:3)
     1,CM(MAXLOC),POSIT(MAXLOC),PXKIK(MAXLOC,0:2),THETA(MAXLOC)
     2,TRANP(0:3,0:3,MAXREL,MAXLOC)
     3,PXKIKE(MAXLOC,0:2),PCXIKE(MAXLOC,0:2),BETA(MAXLOC,0:3)
     4,PIBDE(MAXSIB,MAXLOC,0:2),PXCE(MAXSIB,MAXLOC)
     5,PROBCE(MAXSIB,MAXLOC),PVALUE(MAXSIB,MAXLOC)
     6,SPROBC(MAXSIB,MAXLOC),GCUT(MAXLOC,MAXALL)
     7,PORDER(NSIM)
      INTEGER GENES(MXPTOT,MAXLOC,2),ICHROM(MAXLOC),INPERM(MAXLOC)
     1,NALL(MAXLOC),PERM(MAXLOC),NGEN(MAXLOC),SGENO(MXPTOT,MAXLOC,NHAP)
     2,NISIB(MAXSIB),NJSIB(MAXSIB),PORIG(NSIM)
      LOGICAL DOSIM,WRTMAP,WRTPED
      DATA KCOM,KIN,KOUT,KLOC,KPED,KPUT,KRES/1,5,6,7,8,11,12/
      DATA MINTYP/1/
C
C
C **********************************************************************
C INPUT THE PEDIGREE AND LOCUS DATA.  THESE FILES MUST BE IN SIBMED 
C FORMAT.
C **********************************************************************
C
C INPUT THE FILE NAMES AND OPEN THE FILES.
C
      CALL FOPEN(DOSIM,ECUT,ERROR,ISEED1,ISEED2,ISEED3,KCOM,KLOC,KIN
     1,KOUT,KPED,KPUT,KRES,PCUT,WRTMAP,WRTPED,XXSIGN,XYSIGN)
C
C CHECK FLEXIBLE CODE PARAMETERS FOR PROGRAM DESIGN LIMITS.
C
      CALL CHKPAR(KOUT,MAXALL,MAXLOC)
C
C CHECK FOR VALID VALUES OF VARIABLES SPECIFIED IN CONTROL FILE AND
C ISSUE WARNINGS IF NECESSARY.
C
      CALL CHKVAL(DOSIM,ECUT,ERROR,ISEED1,ISEED2,ISEED3,KOUT,NSIM,PCUT)
C
C READ THE LOCUS FILE, ESTABLISH THE ALLELE MATCHES, AND READ THE
C LOCUS POSITIONS.  CHECK FOR LOCUS FILE ERRORS.
C
      CALL LOCFIL(ALLFRQ,ALLNAM,CM,ICHROM,INPERM,KLOC,KOUT,KRES
     1,LOCNAM,MAXALL,MAXLOC,MAXREL,NALL,NFRQPR,NLOCI,NLOCIN,PERM
     2,POSIT,THETA,TRANP,TRMODE,WRTMAP)
C
C READ THE PEDIGREE FILE AND CHECK FOR ERRORS.
C
      CALL PEDFIL(ALLNAM,CGENO,GENES,ID,INPERM,KOUT,KPED
     1,KRES,LOCNAM,MAXALL,MAXFAM,MAXLOC,MAXPEO,MAXREL,MINTYP
     2,MXPTOT,NALL,NFXTOT,NLOCI,NLOCIN,NPAIR,NPED,NRNTOT,P1ID
     3,P2ID,PEDID,SEX,TWIN,TRANP,WRTPED,XXSIGN,XYSIGN)
C
C
C **********************************************************************
C PERFORM THE ACTUAL ANALYSIS, CALCULATING THE POSTERIOR PROBABILITY OF
C MARKER ERROR OR MUTATION FOR EACH SIBLING PAIR.
C **********************************************************************
C
C
      CALL ANALYZ(ALLFRQ,ALPHA,BETA,ERROR,GENES,KOUT,KRES,LOCNAM,MAXALL
     1,MAXFAM,MAXLOC,MAXREL,MAXSIB,MINREL,MXPTOT,NALL,NFXTOT,NISIB,NJSIB
     2,NLOCI,NPAIR,NRNTOT,P1ID,P2ID,PCXIKE,PEDID,PIBDE,PROBCE,PXCE,PXKIK
     3,PXKIKE,TRANP,TWIN)
C
C
C
C **********************************************************************
C EITHER (1) PERFORM THE SIMULATIONS IN ORDER TO DETERMINE THE CUTOFF
C CONSTANTS FOR THE OSBSERVED POSTERIOR ERROR PROBABILITIES OR (2) 
C PRODUCE A LIST OF THE OBSERVED POSTERIOR ERROR PROBABILITIES ABOVE
C A SPECIFIED CUTOFF.
C **********************************************************************
C
C
      IF(.NOT.DOSIM)GO TO 10
C
      CALL SIMCUT(ALLFRQ,ALPHA,BETA,ERROR,GCUT,ISEED1,ISEED2,ISEED3
     1,KOUT,MAXALL,MAXLOC,MAXPEO,MAXREL,MAXSIB,MXPTOT,NALL,NGEN,NHAP
     2,NLOCI,NPAIR,NSIM,PCXIKE,PIBDE,PORDER,PORIG,PROBCE,PVALUE,PXCE
     3,PXKIK,PXKIKE,SGENO,SPROBC,THETA,TRANP)
C
C
C **********************************************************************
C REPORT SIB PAIR-MARKER COMBINATIONS LIKELY TO BE INCORRECT.
C **********************************************************************
C
 10   CALL REPORT(CM,DOSIM,ECUT,ID,KPUT,KRES,LOCNAM,MAXLOC,MAXSIB,MXPTOT
     1,NISIB,NJSIB,NLOCI,NPAIR,PCUT,PEDID,PROBCE,PVALUE)
C
C
      STOP
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE FOPEN(DOSIM,ECUT,ERROR,ISEED1,ISEED2,ISEED3,KCOM,KLOC
     1,KIN,KOUT,KPED,KPUT,KRES,PCUT,WRTMAP,WRTPED,XXSIGN,XYSIGN)
C
C READ IN THE FILE NAMES, OPEN THE FILES AND READ VALUES IN THE
C CONTROL FILE.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER*1 ECHMAP,ECHPED,SIM
      CHARACTER*8 XXSIGN,XYSIGN
      CHARACTER*72 CONFN,FILNAM,LOCFN,PEDFN,RESFN,OUTFN
      LOGICAL DOSIM,WRTMAP,WRTPED
C
      WRITE(KOUT,101)
 101  FORMAT(A)
      WRITE(KOUT,102)
 102  FORMAT(' ***************************************************** ')
      WRITE(KOUT,103)
 103  FORMAT(' ************** SIBMED.FOR VERSION 1.0 *************** ')
      WRITE(KOUT,104)
 104  FORMAT(' ************** WRITTEN BY J.A. DOUGLAS ************** ')
      WRITE(KOUT,105)
 105  FORMAT(' ******************* APRIL 18, 2000 ****************** ')
      WRITE(KOUT,106)
 106  FORMAT(' ***************************************************** ')
      WRITE(KOUT,101)
C
 10   WRITE(KOUT,101)' CONTROL FILE NAME:  '
      READ(KIN,101,ERR=10)CONFN
      FILNAM=CONFN
      OPEN(KCOM,FILE=FILNAM,STATUS='OLD',ERR=80)

      READ(KCOM,101,ERR=90)LOCFN
      FILNAM=LOCFN
      OPEN(KLOC,FILE=FILNAM,STATUS='OLD',ERR=80)
C
      READ(KCOM,101,ERR=90)PEDFN
      FILNAM=PEDFN
      OPEN(KPED,FILE=FILNAM,STATUS='OLD',ERR=80)
C
      READ(KCOM,101,ERR=90)RESFN
      FILNAM=RESFN
      OPEN(KRES,FILE=FILNAM,STATUS='UNKNOWN',ERR=80)
C
      READ(KCOM,101,ERR=90)OUTFN
      FILNAM=OUTFN
      OPEN(KPUT,FILE=FILNAM,STATUS='UNKNOWN',ERR=80)
C
      WRITE(KRES,101)
      WRITE(KRES,102)
      WRITE(KRES,103)
      WRITE(KRES,104)
      WRITE(KRES,105)
      WRITE(KRES,106)
C
      WRITE(KRES,107)LOCFN
 107  FORMAT(/' LOCUS FILE:   ',A48/) 
      WRITE(KRES,108)PEDFN
 108  FORMAT(' PEDIGREE FILE:   ',A48/) 
      CALL DRAWLN(KRES,1)
C
      READ(KCOM,101,ERR=90)ECHMAP
      CALL MBLANK(ECHMAP)
      WRTMAP=.FALSE.
      IF(ECHMAP(1:1).EQ.'Y'.OR.ECHMAP(1:1).EQ.'y')WRTMAP=.TRUE.
C
      READ(KCOM,101,ERR=90)ECHPED
      CALL MBLANK(ECHPED)
      WRTPED=.FALSE.
      IF(ECHPED(1:1).EQ.'Y'.OR.ECHPED(1:1).EQ.'y')WRTPED=.TRUE.
C
      READ(KCOM,101,ERR=90)XXSIGN
      READ(KCOM,101,ERR=90)XYSIGN
C
      READ(KCOM,109,ERR=90)ERROR
C
      WRITE(KRES,111)ERROR
 111  FORMAT(' PRIOR ERROR RATE =     ',F7.4/)
C
      READ(KCOM,101,ERR=90)SIM
      CALL MBLANK(SIM)
      DOSIM=.FALSE.
C
      IF(SIM(1:1).EQ.'Y'.OR.SIM(1:1).EQ.'y')THEN 
         DOSIM=.TRUE.
         READ(KCOM,109,ERR=90)PCUT
         READ(KCOM,110,ERR=90)ISEED1
         READ(KCOM,110,ERR=90)ISEED2
         READ(KCOM,110,ERR=90)ISEED3
C
         WRITE(KRES,112)PCUT
 112     FORMAT(' PRESCRIBED FALSE POSITIVE RATE = ',F7.4/)
         CALL DRAWLN(KRES,1) 
C
      ELSE
         READ(KCOM,109,ERR=90)ECUT
C
         WRITE(KRES,114)ECUT
 114     FORMAT(' POSTERIOR ERROR PROBABILITY CUTOFF = ',F7.4/)
         CALL DRAWLN(KRES,1)
C
      END IF
C
C
 109  FORMAT(D8.6)
 110  FORMAT(I5)  
C
C      
      WRITE(KOUT,101)' ANALYZING THE SIB PAIRS ... '
C      
C    
      RETURN
C
C
   80 CALL MBLANK(FILNAM)
      I=INDEX(FILNAM,' ')
      IF(I.EQ.0)I=1
      WRITE(KOUT,113)FILNAM(1:I-1)
 113  FORMAT(' *** ERROR *** UNABLE TO OPEN FILE ',A,'.')
      STOP
C
  90  WRITE(KOUT,115)
 115  FORMAT(' *** ERROR *** UNABLE TO READ PART OF CONTROL'
     1,' FILE.  VERIFY THAT'/15X,'THE CONTROL FILE CONSISTS'
     2,' OF ELEVEN OR FOURTEEN ENTRIES WITH'/15X,'FORMATS AS IN THE'
     3,' DOCUMENTATION.')
      STOP
C
C
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE DRAWLN(KDEV,NRET)
C
C DRAW A SEPARATING LINE IN THE SPECIFIED DEVICE.  INCLUDE
C NRET CARRIAGE RETURNS AT THE END.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
      WRITE(KDEV,101)
 101  FORMAT(1X,76('_'))
C
      IF(NRET.LE.0)GO TO 20
C
      DO 10 IRET=1,NRET
        WRITE(KDEV,*)
 10   CONTINUE
C
 20   RETURN
      END      
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE MBLANK(STRING)
C
C THIS SUBROUTINE MOVES ALL BLANKS IN A CHARACTER STRING TO
C THE RIGHT END OF THE STRING.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER C*1,STRING*(*)
C
      I=1
      LSTR=LEN(STRING)
      DO 20 J=1,LSTR
        IF(STRING(J:J).EQ.' ')GO TO 20
          IF(I.EQ.J)GO TO 10
            C=STRING(I:I)
            STRING(I:I)=STRING(J:J)
            STRING(J:J)=C
 10       I=I+1
 20   CONTINUE 
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CHKPAR(KOUT,MAXALL,MAXLOC)
C
C NOTIFY USER IF PROGRAM LIMITS OF ADJUSTABLE PARAMETERS ARE 
C EXCEEDED OR IF ERROR PROBABILITY IS OUTSIDE [0,1].
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
C
      IF(MAXALL.GT.40)GO TO 10
      IF(MAXLOC.GT.500)GO TO 20
C
      RETURN
C
 10   WRITE(KOUT,101)MAXALL
 101  FORMAT(' *** ERROR *** THE CODE PARAMETER "MAXALL" IS SET'
     1,' TO ',I4,'.'/15X,' IT MUST NOT EXCEED 40.')
      STOP
C
 20   WRITE(KOUT,102)MAXLOC
 102  FORMAT(' *** ERROR *** THE CODE PARAMETER "MAXLOC" IS SET'
     1,' TO ',I6,','/15X,' IT MUST NOT EXCEED 500.')
      STOP
C
C
      END
C
C
C-----------------------------------------------------------------------
      SUBROUTINE CHKVAL(DOSIM,ECUT,ERROR,ISEED1,ISEED2,ISEED3,KOUT,NSIM
     1,PCUT)
C
C CHECK FOR VALID VALUES OF VARIABLES SPECIFIED IN CONTROL FILE
C AND ISSUE WARNINGS IF NECESSARY.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      LOGICAL DOSIM
C
      IF(ERROR.LT.0.D0.OR.ERROR.GT.1.D0)GO TO 30
      IF(ISEED1.GT.32767.OR.ISEED2.GT.32767.OR.ISEED3.GT.32767)
     1 GO TO 40   
C
      IF(DOSIM)THEN
         IF(PCUT.LT.0.D0.OR.PCUT.GT.1.D0)GO TO 60
         IF(PCUT.GT.1.D-3)WRITE(KOUT,101)
 101  FORMAT(' **WARNING**FALSE POSITIVE RATES >.001 ARE NOT '
     1,'RECOMMENDED! ')
         IF(PCUT.LT.(10.D0/NSIM))WRITE(KOUT,102)
 102  FORMAT(' **WARNING**INCREASE THE CODE PARAMETER "NSIM" OR'
     1,' INCREASE THE PRESCRIBED FALSE POSITIVE RATE. ')
C
      ELSE
C
         IF(ECUT.LT.0.D0.OR.ECUT.GT.1.D0)GO TO 50
         IF(ECUT.LT.9.D-1)WRITE(KOUT,105)
 105  FORMAT(' **WARNING**POSTERIOR ERROR CUTOFFS < 0.90 ARE NOT '
     1,'RECOMMENDED! ')
C
      END IF
C
      RETURN
C
 30   WRITE(KOUT,103)
 103  FORMAT(' *** ERROR *** THE PRIOR ERROR RATE MUST'
     1,' BE IN THE INTERVAL [0,1].')
      STOP
C
 40   WRITE(KOUT,104)
 104  FORMAT(' *** ERROR *** THE RANDOM NUMBER SEEDS MUST BE'
     1,' LESS THAN OR EQUAL TO 32,767.')
      STOP
C
 50   WRITE(KOUT,106)
 106  FORMAT(' *** ERROR *** THE POSTERIOR ERROR CUTOFF MUST BE'
     1,' IN THE INTERVAL [0,1].')
      STOP
C
 60   WRITE(KOUT,107)
 107  FORMAT(' *** ERROR *** THE PRESCRIBED FALSE POSITIVE RATE'
     1,' MUST BE IN THE INTERVAL [0,1].')
      STOP
C
C
      END
C
C-----------------------------------------------------------------------
C
      SUBROUTINE LOCFIL(ALLFRQ,ALLNAM,CM,ICHROM,INPERM,KLOC,KOUT
     1,KRES,LOCNAM,MAXALL,MAXLOC,MAXREL,NALL,NFRQPR,NLOCI,NLOCIN
     2,PERM,POSIT,THETA,TRANP,TRMODE,WRTMAP)
C
C READ THE LOCUS FILE AND LOCUS POSITIONS AND ESTABLISH EQUIVALANCES 
C BETWEEN BAND SIZE ALLELES AND NUMBERED 1 TO NALL ALLELES.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER*2 ACHAR
      CHARACTER*4 CCHAR
      CHARACTER*8 ALLNAM(MAXLOC,MAXALL),FCHAR,LOCNAM(MAXLOC),PCHAR
     1,TRMODE(MAXLOC)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),MAPFUN,POSIT(MAXLOC)
     1,THETA(MAXLOC),TRANP(0:3,0:3,MAXREL,MAXLOC),CM(MAXLOC)
      INTEGER ICHROM(MAXLOC),INPERM(MAXLOC),NALL(MAXLOC),PERM(MAXLOC)
     1,PERML
      LOGICAL WRTMAP
C
C FIRST PASS:  WHILE LOCI REMAIN, DETERMINE THE LOCUS POSITIONS.
C NLOCI:  NUMBER OF LOCI TO BE USED IN THE ANALYSIS.
C NLOCIN:  NUMBER OF LOCI INPUT.
C
C INITIALIZE THE COUNTS.
C
      NLOCIN=0
      NLOCI=0
C
C READ THE DATA ON THE NEXT MARKER.
C
 1    READ(KLOC,101,END=40,ERR=200)LOCNAM(NLOCIN+1),TRMODE(NLOCIN+1)
     1,NA,ACHAR,ICHROM(NLOCIN+1),CCHAR,POS,PCHAR
 101    FORMAT(2A8,I2,T17,A2,2X,I4,T21,A4,F8.5,T25,A8)
C
C UPDATE THE COUNTS.
C
        NLOCIN=NLOCIN+1
C
        CM(NLOCIN)=100.0D0*POS
C
        CALL MBLANK(LOCNAM(NLOCIN))
        CALL MBLANK(TRMODE(NLOCIN))
        CALL MBLANK(ACHAR)
        CALL MBLANK(CCHAR)
        CALL MBLANK(PCHAR)
C
C CHECK FOR THE FOLLOWING ERRORS:
C   BLANK LOCUS NAME FIELD.
C   BLANK OR INVALID TRANSMISSION MODE FIELD.
C   # OF ALLELES FIELD BLANK OR OUT OF RANGE.
C
        IF(LOCNAM(NLOCIN)(1:1).EQ.' ')GO TO 230
        IF(TRMODE(NLOCIN).NE.'AUTOSOME'.AND.TRMODE(NLOCIN)
     1    .NE.'X-LINKED')GO TO 250
        IF(ACHAR(1:1).EQ.' ')GO TO 260
        IF(NA.GT.MAXALL)GO TO 270
C
C CHECK AND DEAL WITH CHROMOSOME FIELD.  INTERPRET BLANK
C CHROMOSOME FIELD AS 0 AND WARN IF CHROMOSOME # TOO LARGE.
C SINCE CHROMOSOME FIELD IN LOCUS FILE IS 4 DIGITS WIDE, A
C CHROMOSOME NUMBER OF 10000 OR MORE IS CAUSE FOR WARNING.
C
        IF(CCHAR(1:1).EQ.' ')ICHROM(NLOCIN)=0
        IF(ICHROM(NLOCIN).GE.1.D4)WRITE(KOUT,102)LOCNAM(NLOCIN)
        IF(ICHROM(NLOCIN).GT.0)NLOCI=NLOCI+1
        IF(ICHROM(NLOCIN).LE.0)GO TO 10
C
C CHECK THAT POSITION IS PRESENT AND VALID.
C
          IF(PCHAR(1:1).EQ.' ')GO TO 280
          IF(POS.LT.0.D0.OR.POS.GT.50.D0)GO TO 290
C
C INITIALIZE.  THE LOCI HAVE BEEN READ IN BASED ON THEIR ORDER
C IN THE LOCUS FILE.  PERM IS AN INDEX ARRAY WHICH ALLOWS US TO
C REFER TO THE LOCI IN THEIR PROPER ORDER ALONG THE GENOME.
C INPERM IS AN INDEX ARRAY WHICH ALLOWS US TO GO IN THE REVERSE
C DIRECTION, FROM GENOME ORDER TO ORDER IN THE LOCUS FILE.  THETA
C GIVES ESTIMATED RECOMBINATION FRACTIONS BETWEEN THE GIVEN LOCUS
C AND THE NEXT LOCUS ALONG THE GENOME.
C
 10     PERM(NLOCIN)=NLOCIN
        INPERM(NLOCIN)=NLOCIN
        THETA(NLOCIN)=0.5D0
C
C SET THE POSITION FOR THE CURRENT LOCUS:  POSIT(NLOCIN).
C IF CHROMOSOME AND POSITION ARE SPECIFIED, IT IS 100*CHROM+POSITION.
C IF CHROMOSOME IS ZERO OR NEGATIVE, THIS LOCUS WILL BE IGNORED IN THE
C ANALYSIS.  SINCE THE CHROMOSOME FIELD IN THE LOCUS FILE IS 4 DIGITS
C WIDE, THE LARGEST POSSIBLE CHROMOSOME INDEX IS 9999.  THEREFORE, THE
C LARGEST VALID LOCUS POSITION IS 100*9999+50=999950.  LOCI TO BE IGNORED
C ARE ASSIGNED POSITIONS OF AT LEAST 1000001.
C
        IF(ICHROM(NLOCIN).LE.0)P=1.D6+NLOCIN
        IF(ICHROM(NLOCIN).GT.0)P=ICHROM(NLOCIN)*100.D0+POS
        POSIT(NLOCIN)=P
C
C SKIP THE ALLELE LINES OF THE LOCUS FILE FOR THIS LOCUS.
C
 20     DO 30 IALL=1,NA
          READ(KLOC,103,ERR=210)
 103      FORMAT(A)
 30     CONTINUE
      GO TO 1  
C
C CHECK NUMBER OF LOCI AND VALIDITY OF CHROMOSOME DESIGNATIONS.
C
 40   IF(NLOCIN.GT.MAXLOC)GO TO 300
C
C OUTPUT MARKER MAP IF REQUESTED.
C
      IF(.NOT.WRTMAP)GO TO 50
        WRITE(KOUT,104)
 104    FORMAT(' WRITING MARKER MAP TO RESULTS FILE ... ')
        WRITE(KRES,105)
 105    FORMAT(' *** MARKERS USED IN ANALYSIS *** ')
        CALL DRAWLN(KRES,1)
        WRITE(KRES,106)
 106    FORMAT('   MARKER   CHROM   POSITION ')
C
C DETERMINE THE ORDER OF THE LOCI IN THE GENOME AND THE RECOMBINATION
C FRACTIONS BETWEEN THEM.
C
 50   CALL INDEXX(POSIT,PERM,NLOCIN,MAXLOC)
        ICHRPR=100000
        DO 70 LOCUS=1,NLOCIN
          PERML=PERM(LOCUS)
          ICHR=ICHROM(PERML)
          IF(.NOT.WRTMAP)GO TO 60
            IF(ICHR.NE.ICHRPR)CALL DRAWLN(KRES,1)
            IF(ICHR.GT.0)WRITE(KRES,107)LOCNAM(PERML),ICHR
     1      ,DMOD(POSIT(PERML),100.D0)
 107        FORMAT(2X,A8,3X,I2,5X,F6.3)
 60       DIFF=POSIT(PERM(LOCUS+1))-POSIT(PERML)
          IF(DIFF.LT.50.D0)THETA(LOCUS)=MAPFUN(DIFF)
          ICHRPR=ICHR
C
 70     CONTINUE
C
C
C SET THE INVERSE PERMUTATION.  INITIALIZE THE TRANSITION 
C PROBABILITY LOOKUP TABLE.
C
      DO 160 LOCUS=1,NLOCIN-1
C
        INPERM(PERM(LOCUS))=LOCUS
C
        DO 110 I=0,3
          DO 100 J=0,3
            DO 90 K=1,MAXREL
              TRANP(I,J,K,LOCUS)=0.D0
 90         CONTINUE
 100      CONTINUE
 110    CONTINUE
C
C FOR ASSUMED LINKAGE, USE THE PREVIOUSLY CALCULATED ESTIMATES
C OF THETA.  TRANP IS THE IBD TRANSITION PROBABILITY ARRAY FOR
C THE SIBLING PAIR UNDER CONSIDERATION.  IT IS INDEXED BY CURRENT 
C STATE, NEXT STATE, SIB CLASS, AND LOCUS.
C
C KEY TO INDICES:
C
C CURRENT STATE AND NEXT STATE:
C   0=(0,0) -- NEITHER ALLELE SHARED IBD BY THE RELATIVE PAIR.
C   1=(0,1) -- FIRST ALLELE NOT SHARED IBD, SECOND SHARED IBD.
C   2=(1,0) -- FIRST ALLELE SHARED IBD, SECOND NOT SHARED IBD.
C   3=(1,1) -- BOTH ALLELES SHARED IBD.
C
C RELATIVE CLASS:
C   1=FULL SIBS
C   2=HALF SIBS
C
C LOCUS:
C   THE MARKER AT WHICH IBD SHARING IS BEING MEASURED.
C
        TH=THETA(LOCUS)
        PSI=TH*TH+(1.D0-TH)**2
        PSI1=1.D0-PSI
        PSI2=PSI*PSI
        PSI12=PSI1*PSI1
        PSIMIX=PSI*PSI1
C
        DO 150 I=0,3
          DO 140 J=0,3
            IF(I.NE.J)GO TO 120
              TRANP(I,J,1,LOCUS)=PSI2
              GO TO 140
 120        IF(I+J.NE.3)GO TO 130
              TRANP(I,J,1,LOCUS)=PSI12
              GO TO 140
 130        TRANP(I,J,1,LOCUS)=PSIMIX
 140      CONTINUE
 150    CONTINUE
C
        TRANP(0,0,2,LOCUS)=PSI
        TRANP(0,1,2,LOCUS)=PSI1
        TRANP(1,0,2,LOCUS)=PSI1
        TRANP(1,1,2,LOCUS)=PSI
C
 160  CONTINUE
C 
C RE-READ THE INFORMATION FOR EACH LOCUS FROM THE LOCUS FILE.  LEFT 
C JUSTIFY EACH ALLELE NAME.  SORT THE LOCUS INFORMATION ACCORDING 
C TO THE PERMUTATION. 
C
      REWIND(KLOC)
      DO 190 LOC=1,NLOCIN
        LRANK=INPERM(LOC)
        READ(KLOC,108,ERR=220)LOCNAM(LRANK),NA
 108    FORMAT(A8,8X,I2)
        CALL MBLANK(LOCNAM(LRANK))
        NALL(LRANK)=NA
        AFTOT=0.D0
        DO 170 IALL=1,NA
          READ(KLOC,109,ERR=220)ALLNAM(LRANK,IALL),ALLFRQ(LRANK,IALL)
     1    ,FCHAR
 109      FORMAT(A8,F8.5,T9,A8)
          CALL MBLANK(ALLNAM(LRANK,IALL))
          CALL MBLANK(FCHAR)
C
C CHECK FOR THE FOLLOWING ERRORS:
C   BLANK ALLELE NAME.
C   BLANK ALLELE FREQUENCY.
C   ALLELE FREQUENCY OUT OF RANGE.
C
          IF(ALLNAM(LRANK,IALL)(1:1).EQ.' ')GO TO 320
          IF(FCHAR.EQ.' ')GO TO 330
          AFRQ=ALLFRQ(LRANK,IALL)
          IF(AFRQ.LE.0.D0.OR.AFRQ.GT.1.D0)GO TO 340
          AFTOT=AFTOT+AFRQ
 170    CONTINUE
C
C IF ALLELE FREQUENCIES AT THIS LOCUS DO NOT SUM TO 1, WARN THE
C USER AND RESCALE ALLELE FREQUENCIES.  THE PARAMETER NFRQPR
C DETERMINES THE NUMBER OF SIGNIFICANT DIGITS TO WHICH WE CARRY
C THIS CALCULATION.
C
        AFSLOP=1.D0
        DO 175 I=1,NFRQPR
          AFSLOP=AFSLOP*1.D-1
 175    CONTINUE
        AFLOW=1.D0-AFSLOP
        AFHI=1.D0+AFSLOP
        IF(AFTOT.GT.AFLOW.AND.AFTOT.LT.AFHI)GO TO 190
          WRITE(KOUT,111)LOCNAM(LRANK),AFTOT
          DO 180 IALL=1,NA
            ALLFRQ(LRANK,IALL)=ALLFRQ(LRANK,IALL)/AFTOT
 180      CONTINUE
C
 190  CONTINUE
C
C DO SOME HOUSEKEEPING:  NO LONGER NEED LOCUS FILE.
C
      CLOSE(KLOC)
C
C
      RETURN
C
 102  FORMAT(' *** WARNING *** FOR LOCUS ',A,' IN THE LOCUS FILE,'
     1,' THE CHROMOSOME'/17X,' NUMBER IS GREATER THAN 9999.  THE,'
     2,' LOCUS FILE FORMAT'/17X,' SPECIFIES THAT CHROMOSOMES SHOULD,'
     3,' BE GIVEN AS'/17X,' FOUR-DIGIT INTEGERS.')
C
 111  FORMAT(' *** WARNING *** FOR LOCUS ',A,' IN THE LOCUS FILE,'
     1,' THE GIVEN ALLELE'/17X,' FREQUENCIES SUM TO ',F8.5,' INSTEAD'
     2,' OF SUMMING TO 1.'/17X,' THE ALLELE FREQUENCIES HAVE BEEN'
     3,' RESCALED ACCORDINGLY.')
C
 200  IF(NLOCIN.GT.0)GO TO 210
        WRITE(KOUT,112)
 112    FORMAT(' *** ERROR *** PROBLEM READING FIRST LINE OF LOCUS'
     1  ,' FILE.')
        STOP
C
 210  WRITE(KOUT,113)LOCNAM(NLOCIN)
 113  FORMAT(' *** ERROR *** PROBLEM READING LOCUS FILE AT OR NEAR'
     1,' LOCUS ',A,'.')
      STOP
C
 220  WRITE(KOUT,113)LOCNAM(LRANK)
      STOP
C
 230  IF(NLOCIN.NE.1)GO TO 240
        WRITE(KOUT,112)
        STOP
C
 240  WRITE(KOUT,114)LOCNAM(NLOCIN-1)
 114  FORMAT(' *** ERROR *** FOR THE LOCUS AFTER ',A,' IN THE LOCUS'
     1,' FILE, THE LOCUS NAME'/15X,' IS OMITTED OR IMPROPERLY'
     2,' SPECIFIED.')
      STOP
C
 250  WRITE(KOUT,115)LOCNAM(NLOCIN)
 115  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE, THE'
     1,' MODE OF TRANSMISSION'/15X,' IS OMITTED OR IMPROPERLY'
     2,' SPECIFIED.  IT MUST BE'/15X,' EITHER "AUTOSOME" OR'
     3,' "X-LINKED".')
      STOP
C
 260  WRITE(KOUT,116)LOCNAM(NLOCIN)
 116  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE, THE'
     1,' NUMBER OF ALLELES'/15X,' IS OMITTED OR IMPROPERLY'
     2,' SPECIFIED.')
      STOP
C
 270  WRITE(KOUT,117)LOCNAM(NLOCIN),NA,MAXALL
 117  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE, THE'
     1,' NUMBER OF ALLELES'/15X,' IS ',I4,', WHICH IS GREATER THAN'
     2,' THE ALLOWABLE LIMIT OF ',I2,'.'//15X,' THIS LIMIT MAY BE'
     3,' SET AS HIGH AS 99 BY MODIFYING THE VALUE'/15X,' OF THE'
     4,' CODE PARAMETER "MAXALL".')
      STOP
C
 280  WRITE(KOUT,118)LOCNAM(NLOCIN)
 118  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE, THE'
     1,' LOCUS POSITION IS'/15X,' ABSENT.')
      STOP
C
 290  WRITE(KOUT,119)LOCNAM(NLOCIN),POS
 119  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE, THE'
     1,' POSITION IS ',F16.8/15X,' IT SHOULD BE BETWEEN 0 AND 50'
     2,' MORGANS.')
      STOP
C
 300  WRITE(KOUT,121)NLOCIN,MAXLOC
 121  FORMAT(' *** ERROR *** THE NUMBER OF LOCI OBSERVED IN THE'
     1,' LOCUS FILE'/15X,' IS ',I6,', WHICH IS GREATER THAN THE'
     2,' ALLOWABLE LIMIT OF ',I5,'.'//15X,' THIS LIMIT MAY BE SET'
     3,' AS HIGH AS 10000 BY MODIFYING THE VALUE'/15X,' OF THE CODE'
     4,' PARAMETER "MAXLOC".')
      STOP
C
C
 320  WRITE(KOUT,123)LOCNAM(LRANK)
 123  FORMAT(' *** ERROR *** THE INFORMATION UNDER LOCUS ',A,' IN THE'
     1,' LOCUS FILE'/15X,' INCLUDES AN IMPROPERLY SPECIFIED ALLELE'
     2,' NAME OR'/15X,' A BLANK LINE.')
      STOP
C
 330  WRITE(KOUT,124)LOCNAM(LRANK),ALLNAM(LRANK,IALL)
 124  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE,'
     1,' ALLELE ',A,' HAS'/15X,' A MISSING OR IMPROPERLY SPECIFIED'
     2,' ALLELE FREQUENCY.')
      STOP
C
 340  WRITE(KOUT,125)LOCNAM(LRANK),ALLNAM(LRANK,IALL),AFRQ
 125  FORMAT(' *** ERROR *** FOR LOCUS ',A,' IN THE LOCUS FILE,'
     1,' ALLELE ',A,' IS'/15X,' ASSIGNED A FREQUENCY OF ',F16.8/15X
     2,' ALLELE FREQUENCIES MUST BE STRICTLY GREATER THAN 0 AND'/15X
     3,' NO LARGER THAN 1.')
      STOP
C
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE INDEXX(ARRAY,INDX,N,MAXN)
C
C INDEX THE ARRAY ARRAY.  THAT IS, CREATE AN ARRAY INDEX SUCH THAT
C ARRAY(INDX(I)) IS IN ASCENDING ORDER FOR THE FIRST N ARRAY ELEMENTS.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION ARRAY(MAXN)
      INTEGER INDX(MAXN)
C
      DO 1 J=1,N
        INDX(J)=J
 1    CONTINUE
C
      L=N/2+1
      IR=N
C
 10   CONTINUE
        IF(L.GT.1)THEN
          L=L-1
          INDXT=INDX(L)
          Q=ARRAY(INDXT)
        ELSE
          INDXT=INDX(IR)
          Q=ARRAY(INDXT)
          INDX(IR)=INDX(1)
          IR=IR-1
          IF(IR.EQ.1)THEN
            INDX(1)=INDXT
            GO TO 30
          ENDIF
        ENDIF
        I=L
        J=L+L
 20     IF(J.LE.IR)THEN
          IF(J.LT.IR.AND.ARRAY(INDX(J)).LT.ARRAY(INDX(J+1)))J=J+1
          IF(Q.LT.ARRAY(INDX(J)))THEN
            INDX(I)=INDX(J)
            I=J
            J=J+J
          ELSE
            J=IR+1
          ENDIF
        GO TO 20
        ENDIF
        INDX(I)=INDXT
      GO TO 10
C
 30   RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      FUNCTION MAPFUN(X)
C
C DETERMINE THE RECOMBINATION FRACTION CORRESPONDING TO MAP DISTANCE
C X MORGANS (100X cM) BY USING KOSAMBI'S MAP FUNCTION.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION MAPFUN
      Y=DEXP(4.D0*X)
      MAPFUN=(Y-1.D0)/(Y+1.D0)/2.D0
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PEDFIL(ALLNAM,CGENO,GENES,ID,INPERM,KOUT
     1,KPED,KRES,LOCNAM,MAXALL,MAXFAM,MAXLOC,MAXPEO,MAXREL
     2,MINTYP,MXPTOT,NALL,NFXTOT,NLOCI,NLOCIN,NPAIR,NPED,NRNTOT
     3,P1ID,P2ID,PEDID,SEX,TWIN,TRANP,WRTPED,XXSIGN,XYSIGN)
C
C READ THE OLD PEDIGREE FILE AND OUTPUT THE NEW PEDIGREE INFORMATION.  
C THE NEW FORMAT WILL HAVE ONLY THE ADEQUATELY DESCRIBED MEMBERS, IT 
C WILL USE THE 1 TO NALL C ALLELE DESIGNATIONS, AND GENOTYPES WILL 
C NOT BE SPLIT BY SLASHES.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER*8 ALLNAM(MAXLOC,MAXALL),ID(MXPTOT),LOCNAM(MAXLOC)
     1,P1ID(MXPTOT),P2ID(MXPTOT),PEDID(MXPTOT),PEDID1,SEX(MXPTOT)
     2,TWIN(MXPTOT),XXSIGN,XYSIGN
      CHARACTER*17 CGENO(MAXPEO,MAXLOC)
      CHARACTER*72 FRMT1,FRMT2
      DOUBLE PRECISION TRANP(0:3,0:3,MAXREL,MAXLOC)
      INTEGER GENES(MXPTOT,MAXLOC,2),INPERM(MAXLOC),NALL(MAXLOC)
      LOGICAL WRTPED
C
C PROMPT THE USER, SINCE THIS STAGE MAY TAKE A WHILE.  OUTPUT
C PEDIGREE FILE IF REQUESTED.
C
      IF(.NOT.WRTPED)GO TO 1
        WRITE(KOUT,101)
 101    FORMAT(' WRITING PEDIGREE INFORMATION TO RESULTS FILE ... ')
        CALL DRAWLN(KRES,1)
        WRITE(KRES,102)
 102    FORMAT(' *** PEDIGREE INFORMATION *** ')
        CALL DRAWLN(KRES,1)
        GO TO 2
 1      WRITE(KOUT,103)' PROCESSING PEDIGREE FILE ... '
 103  FORMAT(A)
C
C READ THE PEDIGREE FORMATS.
C
 2    READ(KPED,103,ERR=90)FRMT1
      READ(KPED,103,ERR=90)FRMT2
C
      CALL MBLANK(FRMT1)
      CALL MBLANK(FRMT2)
C
C OUTPUT FORMATS TO RESULTS FILE IF REQUESTED.  DO SO BEFORE
C ERROR CHECKING AS AN AID TO THE USER IN TRACKING DOWN PROBLEMS.
C
      IF(.NOT.WRTPED)GO TO 3
        WRITE(KRES,103)' INPUT FORMATS: '
        WRITE(KRES,103)FRMT1
        WRITE(KRES,103)FRMT2
        CALL DRAWLN(KRES,1)
C
C CHECK FOR MISSING OR INVALID INPUT FORMATS.
C
      IF(FRMT1(1:1).EQ.' '.OR.FRMT2(1:1).EQ.' ')GO TO 90
      IF(FRMT1(1:1).NE.'('.OR.FRMT2(1:1).NE.'(')GO TO 90
C
C NPED IS THE NUMBER OF FAMILIES CURRENTLY READ IN.  NRNTOT GIVES
C A RUNNING TOTAL OF THE INDIVIDUALS READ IN WHO MEET THE ANALYSIS
C CRITERIA.
C
 3    NPED=0
      NRNTOT=0
C
C WHILE PEDIGREES REMAIN, FIND ALL SIB PAIRS (INCLUDING
C TWINS).  FOR EACH SIB, AT EACH LOCUS, CONVERT THE BAND-LENGTH GENOTYPE
C TO A 1 TO NALL GENOTYPE.  FOR EACH SIB PAIR, WRITE OUT THEIR MODIFIED
C DATA USING THE PERMUTATION SET UP IN LOCFIL.  INCLUDE ONLY THOSE
C LOCI THAT ARE TO BE USED IN THE ANALYSIS.
C
 5    READ(KPED,FRMT1,END=80,ERR=100)NPEO,PEDID1
      CALL MBLANK(PEDID1)
C
      IF(WRTPED)WRITE(KRES,FRMT1)NPEO,PEDID1
C
C UPDATE THE COUNTS.  NFXTOT GIVES THE "FIXED" TOTAL OF INDIVIDUALS
C MEETING THE ANALYSIS CRITERIA UP TO BUT NOT INCLUDING THE CURRENT
C FAMILY.  THUS, NRNTOT-NFXTOT GIVES THE NUMBER OF INDIVIDUALS IN
C THE CURRENT FAMILY WHO MEET THE ANALYSIS CRITERIA.
C
      NFXTOT=NRNTOT
      NPED=NPED+1
      IF(NPED.GT.MAXFAM)GO TO 110
C
C PEDIGREES CAN HAVE AT MOST MAXPEO PEOPLE AND MUST HAVE AT LEAST 1.
C SINCE NPEO=0 IS INVALID, THIS CHECK ALSO HANDLES A BLANK FIELD.
C
      IF(NPEO.LT.1.OR.NPEO.GT.MAXPEO)GO TO 120
C
C SAVE THE GENOTYPES FOR ALL THE MEMBERS IN THIS PEDIGREE WHO HAVE 
C SUFFICIENT DATA PRESENT.
C
      DO 30 IPEO=1,NPEO
        NRNTOT=NRNTOT+1
        IF(NRNTOT.GT.MXPTOT)GO TO 130
        READ(KPED,FRMT2,ERR=140)ID(NRNTOT),P1ID(NRNTOT),P2ID(NRNTOT)
     1  ,SEX(NRNTOT),TWIN(NRNTOT),(CGENO(NRNTOT-NFXTOT,INPERM(LOCUS))
     2  ,LOCUS=1,NLOCIN)
C
        PEDID(NRNTOT)=PEDID1
        CALL MBLANK(ID(NRNTOT))
        CALL MBLANK(P1ID(NRNTOT))
        CALL MBLANK(P2ID(NRNTOT))
        CALL MBLANK(SEX(NRNTOT))
        CALL MBLANK(TWIN(NRNTOT))
C
C OUTPUT PEDIGREE INFORMATION TO RESULTS FILE IF REQUESTED.  DO SO
C BEFORE ERROR CHECKING TO AID THE USER IN TRACKING DOWN PROBLEMS.
C
        IF(WRTPED)WRITE(KRES,FRMT2)ID(NRNTOT),P1ID(NRNTOT)
     1  ,P2ID(NRNTOT),SEX(NRNTOT),TWIN(NRNTOT),(CGENO(NRNTOT-NFXTOT
     2  ,INPERM(LOCUS)),LOCUS=1,NLOCIN)
C
C ALLOW FOR '0' AS MISSING PARENT CODE.
C
        IF(P1ID(NRNTOT).EQ.'0       ')P1ID(NRNTOT)(1:1)=' '
        IF(P2ID(NRNTOT).EQ.'0       ')P2ID(NRNTOT)(1:1)=' '
C
C CHECK FOR THE FOLLOWING ERRORS:
C   MISSING PEDIGREE MEMBER ID.
C   MEMBER ID MATCHES THAT OF OWN PARENT.
C   INVALID SEX CODE.
C
        IF(ID(NRNTOT)(1:1).EQ.' ')GO TO 150
        IF(ID(NRNTOT).EQ.P1ID(NRNTOT).OR.ID(NRNTOT).EQ.P2ID(NRNTOT))
     1    GO TO 160
        IF(SEX(NRNTOT).NE.XXSIGN.AND.SEX(NRNTOT).NE.XYSIGN.AND
     1    .SEX(NRNTOT)(1:1).NE.' ')GO TO 170
C
C NEED BOTH PARENTS TO BE GIVEN.
C
          IF(P1ID(NRNTOT)(1:1).NE.' '.AND.P2ID(NRNTOT)(1:1).NE.' ')
     1      GO TO 10
          NRNTOT=NRNTOT-1
          GO TO 30
 10     NTYPE=0
        ICUR=NRNTOT-NFXTOT
        DO 20 LOCUS=1,NLOCI
          CALL MBLANK(CGENO(ICUR,LOCUS))
          IF(CGENO(ICUR,LOCUS)(1:1).EQ.' ')GO TO 20
            NTYPE=NTYPE+1
 20     CONTINUE
        IF(NTYPE.LT.MINTYP)NRNTOT=NRNTOT-1
 30   CONTINUE
C
      NFMCUR=NRNTOT-NFXTOT
      IF(NFMCUR.NE.1)GO TO 40
          NRNTOT=NRNTOT-1
          GO TO 5
 40   IF(NFMCUR.EQ.0)GO TO 5
C
C WRITE OUT THE PERTINENT INFO INTO THE GENES ARRAY.
C
      DO 70 ISIB=NFXTOT+1,NRNTOT
        DO 60 LOCUS=1,NLOCI
          CALL GETGEN(ALLNAM,ICUR,KOUT,LOCNAM,LOCUS,MAXALL
     1    ,MAXLOC,NALL,NEW1,NEW2,PEDID1,CGENO(ISIB-NFXTOT,LOCUS))
          GENES(ISIB,LOCUS,1)=NEW1
          GENES(ISIB,LOCUS,2)=NEW2
 60     CONTINUE
 70   CONTINUE
C
      GO TO 5
C
C DO SOME HOUSEKEEPING:  NO LONGER NEED PEDIGREE FILE.
C
 80   IF(WRTPED)CALL DRAWLN(KRES,1)
      CLOSE(KPED)  
C
      RETURN
C
 90   WRITE(KOUT,104)
 104  FORMAT(' *** ERROR *** PROBLEM READING FORMAT STATEMENTS'
     1,' AT TOP OF'/15X,' PEDIGREE FILE.')
      STOP
C
 100  WRITE(KOUT,105)NPED+1
 105  FORMAT(' *** ERROR *** IN PEDIGREE NUMBER ',I6,' IN THE'
     1,' PEDIGREE FILE,'/15X,' THERE IS A PROBLEM READING'
     2,' INDIVIDUAL NUMBER,'/15X,' AND/OR PEDIGREE NAME ENTRIES'
     3,' AT THE BEGINNING,'/15X,' OF THE PEDIGREE.')
      STOP
C
 110  WRITE(KOUT,106)MAXFAM
 106  FORMAT(' *** ERROR *** THE NUMBER OF FAMILIES IN THE'
     1,' PEDIGREE FILE EXCEEDS,'/15X,' THE MAXIMUM LIMIT OF '
     2,I6,'.  TO INCREASE THIS LIMIT,'/15X,' MODIFY THE CODE'
     3,' PARAMETER "MAXFAM".')
      STOP
C
 120  WRITE(KOUT,107)NPED,NPEO,MAXPEO
 107  FORMAT(' *** ERROR *** IN PEDIGREE NUMBER ',I6,' IN THE'
     1,' PEDIGREE FILE,'/15X,' THE NUMBER OF PEOPLE SPECIFIED'
     2,' IS ',I6,', WHICH'/15X,' IS OUT OF RANGE.  THE CURRENT'
     3,' UPPER LIMIT'/15X,' IS ',I4,'.'//15X,' THIS LIMIT MAY BE'
     4,' SET HIGHER BY MODIFYING THE VALUE'/15X,' OF THE CODE'
     5,' PARAMETER "MAXPEO".')
      STOP
C
 130  WRITE(KOUT,108)MXPTOT
 108  FORMAT(' *** ERROR *** THERE ARE MORE THAN ',I6,' INDIVIDUALS'
     1,' IN THE DATA SET.'/' THE CODE PARAMETER "MXPTOT" NEEDS TO'
     2,' BE INCREASED.')
      STOP
C
 140  WRITE(KOUT,109)NPED,NRNTOT-NFXTOT
 109  FORMAT(' *** ERROR *** IN PEDIGREE NUMBER ',I6,' IN THE'
     1,' PEDIGREE FILE,'/15X,' THERE IS A PROBLEM READING PERSON'
     2,' NUMBER ',I4,'.')
      STOP
C
 150  WRITE(KOUT,111)NPED,NRNTOT-NFXTOT
 111  FORMAT(' *** ERROR *** IN PEDIGREE NUMBER ',I6,' IN THE'
     1,' PEDIGREE FILE,'/15X,' PERSON NUMBER ',I4,' IS MISSING'
     2,' ITS MEMBER IDENTIFIER.')
      STOP
C
 160  WRITE(KOUT,112)NPED,NRNTOT-NFXTOT
 112  FORMAT(' *** ERROR *** IN PEDIGREE NUMBER ',I6,' IN THE'
     1,' PEDIGREE FILE,'/15X,' PERSON NUMBER ',I4,' HAS A PARENT'
     2,' WHICH MATCHES,'/15X,' THE MEMBER IDENTIFIER.')
      STOP
C
 170  WRITE(KOUT,113)NPED,NRNTOT-NFXTOT,SEX(NRNTOT),XXSIGN,XYSIGN
 113  FORMAT(' *** ERROR *** IN PEDIGREE NUMBER ',I6,' IN THE'
     1,' PEDIGREE FILE,'/15X,' PERSON NUMBER ',I4,' HAS SEX CODE'
     2,' "',A8,'", WHICH'/15X,' IS INVALID.  SEX CODES MUST BE'
     3,' "',A8,'" OR "',A8,'".')
      STOP
C
C
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE GETGEN(ALLNAM,IJSIB,KOUT,LOCNAM,LOCUS,MAXALL
     1,MAXLOC,NALL,NEW1,NEW2,PEDID,GTYPE)
C	
C THIS SUBROUTINE CONVERTS A BAND-LENGTH GENOTYPE INTO A 1 TO NALL
C GENOTYPE.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      INTEGER KALL(2),NALL(MAXLOC)
      CHARACTER*8 ALLNAM(MAXLOC,MAXALL),C(2),GTYPE,LOCNAM(MAXLOC)
     1,PEDID
C
C IF THIS GENOTYPE IS MISSING, RETURN WITH A GENOTYPE OF 00.
C
      NEW1=0
      NEW2=0
      CALL MBLANK(GTYPE)
      IF(GTYPE(1:1).EQ.' ')GO TO 30
C
C LOCATE THE SEPARATING SLASH IN THE ORIGINAL GENOTYPE.
C
      I=INDEX(GTYPE,'/')
      LGTYPE=LEN(GTYPE)
      IF(I.LE.1.OR.I.GE.LGTYPE)GO TO 40
        C(1)=GTYPE(1:MIN(8,I-1))
        C(2)=GTYPE(I+1:MIN(I+8,LGTYPE))
C
C IDENTIFY THE TWO ALLELES IN THE ORIGINAL GENOTYPE.
C
        DO 20 I=1,2
          DO 10 J=1,NALL(LOCUS)
            IF(C(I).NE.ALLNAM(LOCUS,J))GO TO 10
              KALL(I)=J
              GO TO 20
 10       CONTINUE
          GO TO 50
 20     CONTINUE
C
        NEW1=KALL(1)
        NEW2=KALL(2)
 30     RETURN
C
C IF AN ALLELE ISN'T FOUND OR THERE IS NO SLASH IN THE GENOTYPE, 
C OUTPUT AN ERROR MESSAGE AND STOP.
C
 40   WRITE(KOUT,101)GTYPE,LOCUS,IJSIB,PEDID
 101  FORMAT(' *** ERROR *** NO SLASH PRESENT IN GENOTYPE ',A8,' AT'
     1,' LOCUS NUMBER ',I4/' FOUND IN DESCENDENT NUMBER ',I4,' IN '
     2,' PEDIGREE ',A8/,'.')
      STOP
C
 50   WRITE(KOUT,102)C(I),LOCUS,IJSIB,PEDID
 102  FORMAT(' *** ERROR *** ALLELE ',A8,' AT LOCUS NUMBER ',I4
     1,' FOUND IN DESCENDENT'/15X,' NUMBER ',I4,' IN PEDIGREE '
     2,A8,' IS NOT IN'/15X,' THE LOCUS FILE.')
      STOP
C
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE ANALYZ(ALLFRQ,ALPHA,BETA,ERROR,GENES,KOUT,KRES,LOCNAM
     1,MAXALL,MAXFAM,MAXLOC,MAXREL,MAXSIB,MINREL,MXPTOT,NALL,NFXTOT
     2,NISIB,NJSIB,NLOCI,NPAIR,NRNTOT,P1ID,P2ID,PCXIKE,PEDID,PIBDE
     3,PROBCE,PXCE,PXKIK,PXKIKE,TRANP,TWIN)
C
C EXAMINE ALL MEMBER PAIRS AND PERFORM THE ANALYSIS ON 
C THOSE PAIRS WHICH MEET THE PROPER CRITERIA.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER*8 LOCNAM(MAXLOC),P1ID(MXPTOT)
     1,P2ID(MXPTOT),PAR1,PAR2,PAR3,PAR4,PEDID(MXPTOT),TWIN(MXPTOT)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),ALPHA(MAXLOC,0:3)
     1,PXKIK(MAXLOC,0:2),TRANP(0:3,0:3,MAXREL,MAXLOC)
     2,PXKIKE(MAXLOC,0:2),PCXIKE(MAXLOC,0:2),BETA(MAXLOC,0:3)
     3,PIBDE(MAXSIB,MAXLOC,0:2),PXCE(MAXSIB,MAXLOC)
     4,PROBCE(MAXSIB,MAXLOC)
      INTEGER GENES(MXPTOT,MAXLOC,2),SIBTYP,NISIB(MAXSIB)
     1,NJSIB(MAXSIB)
C
C INITIALIZE.
C
      NPAIR=0
      SIBTYP=2
C
      WRITE(KOUT,101)NRNTOT
 101  FORMAT(I6,' INDIVIDUALS MEET ANALYSIS CRITERIA ...')
C
      DO 50 ISIB=1,NRNTOT-1
        DO 40 JSIB=ISIB+1,NRNTOT
C
C WITHIN FAMILIES, ONLY ANALYZE THOSE PAIRS WITH BOTH MATCHING PARENTS
C I.E. ONLY FULL SIBS.  DO NOT ANALYZE MZ TWINS AS A PAIR.
C
          IF(PEDID(ISIB).NE.PEDID(JSIB))GO TO 40
          IF(TWIN(ISIB)(1:1).NE.' '.AND.TWIN(ISIB).EQ.TWIN(JSIB))
     1      GO TO 40
C
            PAR1=P1ID(ISIB)
            PAR3=P1ID(JSIB)
            PAR2=P2ID(ISIB)
            PAR4=P2ID(JSIB)
C
              IF(PAR1.NE.PAR3.AND.PAR1.NE.PAR4.AND
     1        .PAR2.NE.PAR3.AND.PAR2.NE.PAR4)GO TO 40
              IF(PAR1.EQ.PAR3.AND.PAR2.EQ.PAR4.OR.PAR1.EQ
     1        .PAR4.AND.PAR2.EQ.PAR3) SIBTYP=1
            IF(SIBTYP.EQ.2)GO TO 40
C
            NPAIR=NPAIR+1
            NISIB(NPAIR)=ISIB
            NJSIB(NPAIR)=JSIB
            IF (NPAIR.GT.MAXSIB) GO TO 80 
C
C ANALYZE THE PAIR INDEXED BY ISIB AND JSIB:
C
C CALCULATE THE PROBABILITY OF CORRECT MARKER DATA FOR BOTH
C MEMBERS OF THE SIBLING PAIR CONDITIONAL ON LINKED MARKER DATA
C AND ERROR RATE E.
C
      CALL PRLINK(ALLFRQ,ALPHA,ERROR,GENES
     1,ISIB,JSIB,MAXALL,MAXLOC,MAXREL,MAXSIB
     2,MXPTOT,NLOCI,NPAIR,PXKIK,SIBTYP,TRANP
     3,PXKIKE,PCXIKE,BETA,PIBDE,PXCE,PROBCE)
C
C
 40     CONTINUE
 50   CONTINUE
C
C  
C REPORT WHEN ALL PAIRS ANALYZED.
C
      WRITE(KOUT,103)NPAIR
  103 FORMAT(I10,' PAIRS COMPLETED ...')
C
C
      RETURN
C
   80 WRITE(KOUT,104) MAXSIB
  104 FORMAT( ' *** ERROR *** THE CODE PARAMETER "MAXSIB" IS SET'
     1,' TO ',I3,'.'/15X,' IT MUST BE INCREASED.')
      STOP
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE PRLINK(ALLFRQ,ALPHA,ERROR,GENES,ISIB1
     1,ISIB2,MAXALL,MAXLOC,MAXREL,MAXSIB
     2,MXPTOT,NLOCI,NPAIR,PXKIK,SIBTYP,TRANP
     3,PXKIKE,PCXIKE,BETA,PIBDE,PXCE,PROBCE)
C
C RUN BAUM'S FORWARD ALGORITHM TO CALCULATE THE JOINT PROBABILITY OF
C THE MARKER DATA FOR THE PAIR USING LINKED MARKERS.  FOR EACH
C LOCUS IN TURN STARTING AT THE FIRST, CALCULATE BY RECURSION ALPHA:
C THE JOINT PROBABILITY OF THE MARKER DATA ON ALL PREVIOUS MARKERS AND
C THE IBD STATUS VECTOR AT THE CURRENT MARKER.
C
C RUN BAUM'S BACKWARD ALGORITHM TO CALCULATE BETA:  THE CONDITIONAL
C PROBABILITY OF THE MARKER DATA FROM THE (CURRENT+1) MARKER TO THE
C LAST MARKER GIVEN THE IBD STATUS VECTOR AT THE CURRENT MARKER.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),ALPHA(MAXLOC,0:3)
     1,PXKIK(MAXLOC,0:2),PXKIKE(MAXLOC,0:2),PCXIKE(MAXLOC,0:2)
     2,BETA(MAXLOC,0:3),PIBDE(MAXSIB,MAXLOC,0:2),PXCE(MAXSIB,MAXLOC)
     3,PROBCE(MAXSIB,MAXLOC),TRANP(0:3,0:3,MAXREL,MAXLOC)
      INTEGER GENES(MXPTOT,MAXLOC,2),SIBTYP
C
C INITIALIZE.
C
      LOCLIM=NLOCI-1
C
C CALCULATE THE CONDITIONAL PROBABILITY OF THE MARKER DATA
C AT EACH LOCUS GIVEN EACH IBD SHARING STATUS FOR THE PAIR.
C
      CALL CALCPK(ALLFRQ,GENES,ISIB1,ISIB2,MAXALL,MAXLOC,MXPTOT
     1,NLOCI,PXKIK,NPAIR)
C
C
C GIVEN ERROR E, CALCULATE THE CONDITIONAL PROBABILITY OF THE
C MARKER DATA AT EACH LOCUS GIVEN EACH IBD SHARING STATUS FOR
C THE PAIR.  ALSO GIVEN ERROR E, CALCULATE THE CONDITIONAL
C PROBABILITY OF CORRECT MARKER DATA FOR BOTH MEMBERS OF THE SIB
C PAIR AT EACH LOCUS GIVEN EACH IBD SHARING STATUS FOR THE PAIR.
C
      CALL CALPKE(ERROR,ERRORC,MAXLOC,NLOCI,PCXIKE,PXKIK,PXKIKE
     1,NPAIR,ISIB1,ISIB2,GENES,ALLFRQ,MAXALL,MXPTOT)
C
C CALULATE THE INITIAL FORWARD PROBABILITY FOR THE PAIR.
C
      CALL INITA(ALPHA,MAXLOC,SIBTYP)
C
C CALCULATE THE SUCCESSIVE FORWARD PROBABILITIES FOR THE PAIR.
C
      CALL CALFOR(ALPHA,LOCLIM,MAXLOC,MAXREL,NLOCI,PROBXE
     1,PXKIKE,SIBTYP,TRANP,NPAIR)
C
C CALCULATE THE LAST BACKWARD PROBABILITY FOR THE PAIR.
C
      CALL INITB(BETA,MAXLOC,NLOCI)
C
C CALCULATE THE BACKWARD PROBABILITIES FOR THE PAIR.
C
      CALL CALBAC(BETA,LOCLIM,MAXLOC,MAXREL,NLOCI,PXKIKE
     1,SIBTYP,TRANP,NPAIR)
C
C
C CALCULATE THE PROBABILITY OF MARKER MUTATION OR ERROR FOR
C EITHER SIB. 
C
      IF (SIBTYP.EQ.2) GO TO 60
C
C FOR FULL SIBS...
C
      DO 20, I=1,NLOCI
C
      PXCE(NPAIR,I)=ALPHA(I,0)*PCXIKE(I,0)*BETA(I,0)
     1      +ALPHA(I,1)*PCXIKE(I,1)*BETA(I,1)
     2      +ALPHA(I,2)*PCXIKE(I,1)*BETA(I,2)
     3      +ALPHA(I,3)*PCXIKE(I,2)*BETA(I,3)
C
      PROBCE(NPAIR,I)=PXCE(NPAIR,I)/PROBXE
C
      PIBDE(NPAIR,I,0)=(ALPHA(I,0)*PXKIKE(I,0)*BETA(I,0))/PROBXE
      PIBDE(NPAIR,I,1)=(ALPHA(I,1)*PXKIKE(I,1)*BETA(I,1)
     1                 +ALPHA(I,2)*PXKIKE(I,1)*BETA(I,2))/PROBXE
      PIBDE(NPAIR,I,2)=(ALPHA(I,3)*PXKIKE(I,2)*BETA(I,3))/PROBXE
C
   20 CONTINUE
C   
      GO TO 80
C
C
C FOR HALF SIBS...
C
   60 DO 30, I=1,NLOCI
C
      PXCE(NPAIR,I)=ALPHA(I,0)*PCXIKE(I,0)*BETA(I,0)
     1      +ALPHA(I,1)*PCXIKE(I,1)*BETA(I,1)
C
      PROBCE(NPAIR,I)=PXCE(NPAIR,I)/PROBXE
C
      PIBDE(NPAIR,I,0)=(ALPHA(I,0)*PXKIKE(I,0)*BETA(I,0))/PROBXE
      PIBDE(NPAIR,I,1)=(ALPHA(I,1)*PXKIKE(I,1)*BETA(I,1)
     1                 +ALPHA(I,2)*PXKIKE(I,1)*BETA(I,2))/PROBXE
C
   30 CONTINUE
C
   80 RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CALCPK(ALLFRQ,GENES,ISIB1,ISIB2,MAXALL,MAXLOC
     1,MXPTOT,NLOCI,PXKIK,NPAIR)
C
C CALCULATE THE CONDITIONAL PROBABILITY OF THE MARKER DATA
C AT EACH LOCUS GIVEN EACH IBD SHARING STATUS FOR THE PAIR.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),PXKIK(MAXLOC,0:2)
      INTEGER GENES(MXPTOT,MAXLOC,2)
C
C
C
C FOR EACH LOCUS ...
C
      DO 170 LOCUS=1,NLOCI
C
        IG11=GENES(ISIB1,LOCUS,1)
        IG21=GENES(ISIB2,LOCUS,1)
        IF(IG11*IG21.GT.0)GO TO 10
C
C IF THE DATA ARE MISSING FOR EITHER INDIVIDUAL, THE PROBABILITIES ARE
C NECESSARILY ONE AT THIS LOCUS.
C
          PXKIK(LOCUS,0)=1.D0
          PXKIK(LOCUS,1)=1.D0
          PXKIK(LOCUS,2)=1.D0
          GO TO 170
C
 10     IG12=GENES(ISIB1,LOCUS,2)
        IG22=GENES(ISIB2,LOCUS,2)
C
C DIFFERENTIATE AMONG THE 7 DISTINCT CASES:
C   (II II), (II IJ), (II JJ), (II JK), (IJ IJ), (IJ IK), (IJ KL)
C
        IF(IG11.EQ.IG12)GO TO 50
          IF(IG21.EQ.IG22)GO TO 70
C
C BOTH HETERO, SO EITHER (IJ IJ), (IJ IK), OR (IJ KL).  WE
C KNOW THAT THERE ARE AT LEAST TWO DIFFERENT ALLELES HERE,
C AND WE NEED TO IDENTIFY AS MANY AS TWO OTHERS.
C
            IF(IG11.NE.IG21)GO TO 20
              AFRQI=ALLFRQ(LOCUS,IG11)
              AFRQJ=ALLFRQ(LOCUS,IG12)
              IF(IG12.EQ.IG22)GO TO 140
                AFRQK=ALLFRQ(LOCUS,IG22)
                GO TO 150
 20         IF(IG12.NE.IG22)GO TO 30
              AFRQI=ALLFRQ(LOCUS,IG12)
              AFRQJ=ALLFRQ(LOCUS,IG11)
              AFRQK=ALLFRQ(LOCUS,IG21) 
              GO TO 150
 30         IF(IG11.NE.IG22)GO TO 40
              AFRQI=ALLFRQ(LOCUS,IG11)
              AFRQJ=ALLFRQ(LOCUS,IG12)
              IF(IG12.EQ.IG21)GO TO 140
                AFRQK=ALLFRQ(LOCUS,IG21)
                GO TO 150
 40         IF(IG12.NE.IG21)GO TO 160
              AFRQI=ALLFRQ(LOCUS,IG12)
              AFRQJ=ALLFRQ(LOCUS,IG11)
              AFRQK=ALLFRQ(LOCUS,IG22)
              GO TO 150
C
 50     IF(IG21.EQ.IG22)GO TO 90
C
C 1 HOMO, 2 HETERO, SO EITHER (II IJ) OR (II JK).  AGAIN,
C IDENTIFY THE DIFFERENT ALLELES INVOLVED.
C
          AFRQI=ALLFRQ(LOCUS,IG11)
          IF(IG11.NE.IG21)GO TO 60
            AFRQJ=ALLFRQ(LOCUS,IG22)
            GO TO 120
 60       AFRQJ=ALLFRQ(LOCUS,IG21)
          IF(IG11.EQ.IG22)GO TO 120
            AFRQK=ALLFRQ(LOCUS,IG22)
            GO TO 130
C
C 1 HETERO, 2 HOMO, SO EITHER (IJ II) OR (JK II).  NEED TO
C IDENTIFY THE ALLELES IN THESE CASES.
C
 70       AFRQI=ALLFRQ(LOCUS,IG21)
          IF(IG21.NE.IG11)GO TO 80
            AFRQJ=ALLFRQ(LOCUS,IG12)
            GO TO 120
 80       AFRQJ=ALLFRQ(LOCUS,IG11)
          IF(IG21.EQ.IG12)GO TO 120
            AFRQK=ALLFRQ(LOCUS,IG12)
            GO TO 130
C
C BOTH HOMOZYGOUS, SO EITHER (II II) OR (II JJ).  DON'T NEED
C TO IDENTIFY ALLELES IN EITHER CASE.
C
 90     IF(IG11.NE.IG21)GO TO 110
C
C FOR THE OBSERVED CASE, CALCULATE THE PROBABILITIES.  THE IBD STATUS
C 1 AND 2 PROBABILITIES ARE THE SAME, SO REFER TO BOTH BY INDEX 1.
C
C
C CASE 1: II II
C
 100    AFRQ=ALLFRQ(LOCUS,IG11)
        AFRQSQ=AFRQ*AFRQ
        PXKIK(LOCUS,0)=AFRQSQ*AFRQSQ
        PXKIK(LOCUS,1)=AFRQ*AFRQSQ
        PXKIK(LOCUS,2)=AFRQSQ
        GO TO 170
C
C CASE 2: II JJ (ORDERED PAIR)
C
 110    AFRQX=ALLFRQ(LOCUS,IG11)*ALLFRQ(LOCUS,IG21)
        PXKIK(LOCUS,0)=AFRQX*AFRQX
        PXKIK(LOCUS,1)=0.D0
        PXKIK(LOCUS,2)=0.D0
        GO TO 170
C
C CASE 3: II IJ (ORDERED PAIR)
C
 120    AFRQX=AFRQI*AFRQI*AFRQJ
        PXKIK(LOCUS,0)=2.D0*AFRQI*AFRQX
        PXKIK(LOCUS,1)=AFRQX
        PXKIK(LOCUS,2)=0.D0
        GO TO 170
C
C CASE 4: II JK  (ORDERED PAIR)
C
 130    PXKIK(LOCUS,0)=2.D0*AFRQI*AFRQI*AFRQJ*AFRQK
        PXKIK(LOCUS,1)=0.D0
        PXKIK(LOCUS,2)=0.D0
        GO TO 170
C
C CASE 5: IJ IJ
C
 140    AFRQIJ=AFRQI*AFRQJ
        PXKIK(LOCUS,0)=4.D0*AFRQIJ*AFRQIJ
        PXKIK(LOCUS,1)=AFRQIJ*(AFRQI+AFRQJ)
        PXKIK(LOCUS,2)=2.D0*AFRQIJ
        GO TO 170
C
C CASE 6: IJ IK  (ORDERED PAIR)
C
 150    AFRQX=AFRQI*AFRQJ*AFRQK
        PXKIK(LOCUS,0)=4.D0*AFRQI*AFRQX
        PXKIK(LOCUS,1)=AFRQX
        PXKIK(LOCUS,2)=0.D0
        GO TO 170
C
C CASE 7: IJ KL  (ORDERED PAIR)
C
 160    PXKIK(LOCUS,0)=4.D0*ALLFRQ(LOCUS,IG11)*ALLFRQ(LOCUS,IG12)
     1    *ALLFRQ(LOCUS,IG21)*ALLFRQ(LOCUS,IG22)
        PXKIK(LOCUS,1)=0.D0
        PXKIK(LOCUS,2)=0.D0
C
 170  CONTINUE
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CALPKE(ERROR,ERRORC,MAXLOC,NLOCI,PCXIKE,PXKIK
     1,PXKIKE,NPAIR,ISIB1,ISIB2,GENES,ALLFRQ,MAXALL,MXPTOT)
C
C GIVEN ERROR E, CALCULATE THE CONDITIONAL PROBABILITY OF THE
C MARKER DATA AT EACH LOCUS GIVEN EACH IBD SHARING STATUS FOR
C THE PAIR.  ALSO GIVEN ERROR E, CALCULATE THE CONDITIONAL
C PROBABILITY OF CORRECT MARKER DATA FOR BOTH MEMBERS OF THE SIB
C PAIR AT EACH LOCUS GIVEN EACH IBD SHARING STATUS FOR THE PAIR.
C     
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION PXKIK(MAXLOC,0:2), PXKIKE(MAXLOC,0:2)
     1,PCXIKE(MAXLOC,0:2),ALLFRQ(MAXLOC,MAXALL)
      INTEGER GENES(MXPTOT,MAXLOC,2)
C
      ERRORC=1-ERROR
      ESQ=(1-ERROR)**2
      ESQC=1-(1-ERROR)**2
C
      DO 10, I=1,NLOCI
C
       IG11=GENES(ISIB1,I,1)
       IG12=GENES(ISIB1,I,2)
       IG21=GENES(ISIB2,I,1)
       IG22=GENES(ISIB2,I,2)
C
C CHECK FOR MISSING GENOTYPES AT THIS LOCUS.
C
C IF A SIBLING'S GENOTYPE IS MISSING, SET THE GENOTYPE PROBABILITY PXJ 
C TO ONE SO THAT THE POSTERIOR ERROR PROBABILITY EQUALS ZERO AT THIS
C LOCUS.
C
C       
       IF(IG11*IG12.GT.0)THEN
       GO TO 51
       END IF
       PX1=1.D0
       GO TO 52
 51    IF(IG11.EQ.IG12)THEN
          PX1=(ALLFRQ(I,IG11))**2
       ELSE
          PX1=2.D0*ALLFRQ(I,IG11)*ALLFRQ(I,IG12)
       END IF
C
 52    CONTINUE
C
       IF(IG21*IG22.GT.0)THEN
       GO TO 53
       END IF
       PX2=1.D0
       GO TO 54
 53    IF(IG21.EQ.IG22)THEN
          PX2=(ALLFRQ(I,IG21))**2
       ELSE
          PX2=2.D0*ALLFRQ(I,IG21)*ALLFRQ(I,IG22)
       END IF
C
 54    CONTINUE
C
       DO 20, J=0,2
         PXKIKE(I,J)=ESQ*PXKIK(I,J)+ESQC*PX1*PX2
         PCXIKE(I,J)=PXKIK(I,J)*(1-ERROR+ERROR*PX1)*(1-ERROR+ERROR*PX2)
  20  CONTINUE
C
  10  CONTINUE
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE INITA(ALPHA,MAXLOC,SIBTYP)
C
C INITIALIZE THE RECURSIVE PROBABILITIES FOR THE FIRST MARKER TO THE
C PROBABILITIES FOR THE DIFFERENT IBD STATES OF THE PAIR.  MISSING DATA
C DO NOT MATTER HERE.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION ALPHA(MAXLOC,0:3)
      INTEGER SIBTYP
C
C CHOOSE THE APPROPRIATE RELATIVE TYPE.
C
      GO TO (10,20) SIBTYP
C
C FOR FULL SIBS
C
 10     ALPHA(1,0)=0.25D0
        ALPHA(1,1)=0.25D0
        ALPHA(1,2)=0.25D0
        ALPHA(1,3)=0.25D0
        GO TO 50
C
C FOR HALF SIBS (ASSUMING SHARED SECOND PARENT)
C
 20     ALPHA(1,0)=0.50D0
        ALPHA(1,1)=0.50D0
        ALPHA(1,2)=0.00D0
        ALPHA(1,3)=0.00D0
        GO TO 50
C
 50   RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CALFOR(ALPHA,LOCLIM,MAXLOC,MAXREL,NLOCI
     1,PROBXE,PXKIKE,SIBTYP,TRANP,NPAIR)
C
C CALCULATE THE SUCCESSIVE FORWARD PROBABILITIES FOR THE PAIR.
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION ALPHA(MAXLOC,0:3), TRANP(0:3,0:3,MAXREL,MAXLOC)
     1,PXKIKE(MAXLOC,0:2)
      INTEGER SIBTYP
C
C
      IF (SIBTYP.EQ.2) GO TO 60
C
C FOR FULL SIBS...
C
      DO 40 LOCUS=1,LOCLIM
C
C FOR EACH SUBSEQUENT LOCUS LOCUS+1, UPDATE THE RECURSION.   
C TRANP GIVES THE PROBABILITY OF THE IBD STATUS VECTOR AT MARKER
C I GIVEN THE IBD STATUS VECTOR AT MARKER (I-1) FOR EACH RELATIVE
C TYPE (FULL SIBS IN THIS CASE).  TRANP IS DEPENDENT ON THE INDIVIDUAL 
C LOCI ONLY IN TERMS OF RECOMBINATION FRACTIONS.
C
        ALXPX0=ALPHA(LOCUS,0)*PXKIKE(LOCUS,0)
        ALXPX1=ALPHA(LOCUS,1)*PXKIKE(LOCUS,1)
        ALXPX2=ALPHA(LOCUS,2)*PXKIKE(LOCUS,1)
        ALXPX3=ALPHA(LOCUS,3)*PXKIKE(LOCUS,2)
C
        ALSUM0=TRANP(0,0,1,LOCUS)*ALXPX0
     1    +TRANP(1,0,1,LOCUS)*ALXPX1
     2    +TRANP(2,0,1,LOCUS)*ALXPX2
     3    +TRANP(3,0,1,LOCUS)*ALXPX3
        ALPHA(LOCUS+1,0)=ALSUM0
C
        ALSUM1=TRANP(0,1,1,LOCUS)*ALXPX0
     1    +TRANP(1,1,1,LOCUS)*ALXPX1
     2    +TRANP(2,1,1,LOCUS)*ALXPX2
     3    +TRANP(3,1,1,LOCUS)*ALXPX3
        ALPHA(LOCUS+1,1)=ALSUM1
C
C
        ALSUM2=TRANP(0,2,1,LOCUS)*ALXPX0
     1    +TRANP(1,2,1,LOCUS)*ALXPX1
     2    +TRANP(2,2,1,LOCUS)*ALXPX2
     3    +TRANP(3,2,1,LOCUS)*ALXPX3
        ALPHA(LOCUS+1,2)=ALSUM2
C
C
        ALSUM3=TRANP(0,3,1,LOCUS)*ALXPX0
     1    +TRANP(1,3,1,LOCUS)*ALXPX1
     2    +TRANP(2,3,1,LOCUS)*ALXPX2
     3    +TRANP(3,3,1,LOCUS)*ALXPX3
        ALPHA(LOCUS+1,3)=ALSUM3
C
 40   CONTINUE
C
C CALCULATE THE JOINT PROBABILITY OF THE MARKER DATA BY SUMMING THE LAST
C ALPHAS TIMES THE PROBABILITIES OF THE MARKER DATA FOR THE LAST LOCUS
C CONDITIONAL ON THE IBD STATUS FOR THAT LOCUS.
C
      PROBXE=ALPHA(NLOCI,0)*PXKIKE(NLOCI,0)
     1  +ALPHA(NLOCI,1)*PXKIKE(NLOCI,1)
     2  +ALPHA(NLOCI,2)*PXKIKE(NLOCI,1)
     3  +ALPHA(NLOCI,3)*PXKIKE(NLOCI,2)
C
C
      GO TO 80
C
C FOR HALF SIBS...
C
 60   DO 50 LOCUS=1,LOCLIM
C
C FOR EACH SUBSEQUENT LOCUS LOCUS+1, UPDATE THE RECURSION.   
C TRANP GIVES THE PROBABILITY OF THE IBD STATUS VECTOR AT MARKER
C I GIVEN THE IBD STATUS VECTOR AT MARKER (I-1) FOR EACH RELATIVE
C TYPE (HALF SIBS IN THIS CASE).  TRANP IS DEPENDENT ON THE INDIVIDUAL 
C LOCI ONLY IN TERMS OF RECOMBINATION FRACTIONS.
C
        ALXPX0=ALPHA(LOCUS,0)*PXKIKE(LOCUS,0)
        ALXPX1=ALPHA(LOCUS,1)*PXKIKE(LOCUS,1)
C
        ALSUM0=TRANP(0,0,2,LOCUS)*ALXPX0
     1    +TRANP(1,0,2,LOCUS)*ALXPX1
        ALPHA(LOCUS+1,0)=ALSUM0
C
        ALSUM1=TRANP(0,1,2,LOCUS)*ALXPX0
     1    +TRANP(1,1,2,LOCUS)*ALXPX1
        ALPHA(LOCUS+1,1)=ALSUM1
C
C
 50   CONTINUE
C
C CALCULATE THE JOINT PROBABILITY OF THE MARKER DATA BY SUMMING THE LAST
C ALPHAS TIMES THE PROBABILITIES OF THE MARKER DATA FOR THE LAST LOCUS
C CONDITIONAL ON THE IBD STATUS FOR THAT LOCUS.
C
      PROBXE=ALPHA(NLOCI,0)*PXKIKE(NLOCI,0)
     1  +ALPHA(NLOCI,1)*PXKIKE(NLOCI,1)
C
 80   RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE INITB(BETA,MAXLOC,NLOCI)
C
C INITIALIZE BACKWARD PROBABILITIES ...
C
      DOUBLE PRECISION BETA(MAXLOC,0:3)
C
      BETA(NLOCI,0)=1.0D0
      BETA(NLOCI,1)=1.0D0
      BETA(NLOCI,2)=1.0D0
      BETA(NLOCI,3)=1.0D0
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE CALBAC(BETA,LOCLIM,MAXLOC,MAXREL,NLOCI,PXKIKE
     1,SIBTYP,TRANP,NPAIR)
C
C     CALCULATE THE BACKWARD PROBABILITIES FOR THE PAIR.
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION PXKIKE(MAXLOC,0:2), BETA(MAXLOC,0:3)
     1,TRANP(0:3,0:3,MAXREL,MAXLOC)
      INTEGER SIBTYP
C
      IF (SIBTYP.EQ.2) GO TO 60
C
C FOR FULL SIBS ...
C
      DO 10 I=1,LOCLIM
C
C FOR EACH PREVIOUS LOCUS,....
C
      BEXPX0=BETA(NLOCI-I+1,0)*PXKIKE(NLOCI-I+1,0)
      BEXPX1=BETA(NLOCI-I+1,1)*PXKIKE(NLOCI-I+1,1)
      BEXPX2=BETA(NLOCI-I+1,2)*PXKIKE(NLOCI-I+1,1)
      BEXPX3=BETA(NLOCI-I+1,3)*PXKIKE(NLOCI-I+1,2)
C
      BESUM0=BEXPX0*TRANP(0,0,1,NLOCI-I)
     1  +BEXPX1*TRANP(0,1,1,NLOCI-I)
     2  +BEXPX2*TRANP(0,2,1,NLOCI-I)
     3  +BEXPX3*TRANP(0,3,1,NLOCI-I)
      BETA(NLOCI-I,0)=BESUM0
C
      BESUM1=BEXPX0*TRANP(1,0,1,NLOCI-I)
     1  +BEXPX1*TRANP(1,1,1,NLOCI-I)
     2  +BEXPX2*TRANP(1,2,1,NLOCI-I)
     3  +BEXPX3*TRANP(1,3,1,NLOCI-I)
      BETA(NLOCI-I,1)=BESUM1
C
      BESUM2=BEXPX0*TRANP(2,0,1,NLOCI-I)
     1  +BEXPX1*TRANP(2,1,1,NLOCI-I)
     2  +BEXPX2*TRANP(2,2,1,NLOCI-I)
     3  +BEXPX3*TRANP(2,3,1,NLOCI-I)
      BETA(NLOCI-I,2)=BESUM2
C
      BESUM3=BEXPX0*TRANP(3,0,1,NLOCI-I)
     1  +BEXPX1*TRANP(3,1,1,NLOCI-I)
     2  +BEXPX2*TRANP(3,2,1,NLOCI-I)
     3  +BEXPX3*TRANP(3,3,1,NLOCI-I)
      BETA(NLOCI-I,3)=BESUM3
C
   10 CONTINUE
C
      GO TO 80
C
C FOR HALF SIBS...
C
   60 DO 20 I=1,LOCLIM
C
C FOR EACH PREVIOUS LOCUS,....
C
      BEXPX0=BETA(NLOCI-I+1,0)*PXKIKE(NLOCI-I+1,0)
      BEXPX1=BETA(NLOCI-I+1,1)*PXKIKE(NLOCI-I+1,1)
C
      BESUM0=BEXPX0*TRANP(0,0,2,NLOCI-I)
     1  +BEXPX1*TRANP(0,1,2,NLOCI-I)
      BETA(NLOCI-I,0)=BESUM0
C
      BESUM1=BEXPX0*TRANP(1,0,2,NLOCI-I)
     1  +BEXPX1*TRANP(1,1,2,NLOCI-I)
      BETA(NLOCI-I,1)=BESUM1
C
C
   20 CONTINUE
C
   80 RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE SIMCUT(ALLFRQ,ALPHA,BETA,ERROR,GCUT,ISEED1,ISEED2
     1,ISEED3,KOUT,MAXALL,MAXLOC,MAXPEO,MAXREL,MAXSIB,MXPTOT,NALL,NGEN
     2,NHAP,NLOCI,NPAIR,NSIM,PCXIKE,PIBDE,PORDER,PORIG,PROBCE,PVALUE
     3,PXCE,PXKIK,PXKIKE,SGENO,SPROBC,THETA,TRANP)
C
C SIMULATE GENOTYPE DATA FOR 4-PERSON NUCLEAR FAMILIES UNDER
C THE NULL HYPOTHESIS.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),ALPHA(MAXLOC,0:3)
     1,BETA(MAXLOC,0:3),GCUT(MAXLOC,MAXALL),PXKIK(MAXLOC,0:2)
     2,PXKIKE(MAXLOC,0:2),PCXIKE(MAXLOC,0:2)
     3,PIBDE(MAXSIB,MAXLOC,0:2),PXCE(MAXSIB,MAXLOC)
     4,PROBCE(MAXSIB,MAXLOC)
     5,PVALUE(MAXSIB,MAXLOC),SPROBC(MAXSIB,MAXLOC)
     6,TRANP(0:3,0:3,MAXREL,MAXLOC),THETA(MAXLOC)
     7,PORDER(NSIM)
      INTEGER SGENO(MXPTOT,MAXLOC,NHAP),NALL(MAXLOC),NGEN(MAXLOC)
     1,PORIG(NSIM)
C
C      
C CALCULATE CUTPOINTS FOR THE SIMULATION AND OTHER SETUP INFORMATION.
C
      CALL SETUP(ALLFRQ,GCUT,MAXALL,MAXLOC,NALL,NGEN,NLOCI)
C
C SIMULATE SEPARATELY FOR EACH MARKER
C
      WRITE(KOUT,101)
 101  FORMAT(' SIMULATING TO DETERMINE CUTOFF CONSTANTS ...') 
C
      DO 60 ILOC=1,NLOCI
C
       DO 50 JSIM=1,NSIM
C
        CALL SIMGEN(GCUT,SGENO,ISEED1,ISEED2,ISEED3,MAXALL
     1,MAXLOC,MAXPEO,MXPTOT,NHAP,NALL,NLOCI,THETA,ILOC,ERROR)
C
C
        CALL PRLINK(ALLFRQ,ALPHA,ERROR,SGENO,3,4,MAXALL,MAXLOC
     1,MAXREL,MAXSIB,MXPTOT,NLOCI,1,PXKIK,1,TRANP
     2,PXKIKE,PCXIKE,BETA,PIBDE,PXCE,SPROBC)
C
       PORDER(JSIM)=SPROBC(1,ILOC)
       PORIG(JSIM)=JSIM
C
C
 50    CONTINUE
C
       WRITE(KOUT,102)NSIM,ILOC
 102   FORMAT('    ',I8,' REPLICATES FOR LOCUS ',I4,' COMPLETED ... ')
C
       CALL INDEXX(PORDER,PORIG,NSIM,NSIM)
C
C
C DETERMINE THE P-VALUES BASED ON THE EMPIRICAL DISTRIBUTION UNDER
C THE NULL HYPOTHESIS.
C
C
        DO 40 K=1,NPAIR
C
         IF(PROBCE(K,ILOC).LT.PORDER(PORIG(1)))GO TO 90
C
          DO 30 N=1,NSIM           
C
          IF(PROBCE(K,ILOC).GT.PORDER(PORIG(N)))GO TO 30
          PVALUE(K,ILOC)=(1.D0*(N-1.D0))/(1.D0*NSIM)
          GO TO 40
C
 30      CONTINUE
C
          PVALUE(K,ILOC)=1.D0
          GO TO 40
C
 90       PVALUE(K,ILOC)=0.D0
C
 40     CONTINUE
C
C
 60   CONTINUE
C
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE SETUP(ALLFRQ,GCUT,MAXALL,MAXLOC,NALL,NGEN,NLOCI)
C
C CARRY OUT THE SETUP CALCULATIONS FOR THE SIMULATION.
C     
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION ALLFRQ(MAXLOC,MAXALL),GCUT(MAXLOC,MAXALL)
      INTEGER NALL(MAXLOC),NGEN(MAXLOC)
C
C DETERMINE THE NUMBER OF DIPLOID GENOTYPES AT EACH LOCUS.
C
      DO 10 ILOC=1,NLOCI
        NGEN(ILOC)=(NALL(ILOC)*(NALL(ILOC)+1))/2
 10   CONTINUE
C
C DETERMINE THE ALLELE FREQUENCY CUTPOINTS FOR THE SIMULATION.
C
      DO 30 ILOC=1,NLOCI
        NA=NALL(ILOC)
        GCUT(ILOC,1)=ALLFRQ(ILOC,1)
        DO 20 IALL=2,NA-1
          GCUT(ILOC,IALL)=GCUT(ILOC,IALL-1)+ALLFRQ(ILOC,IALL)
 20     CONTINUE
C
C BUG FIX: FORCE GCUT TO BE 1 AT THE LAST CUT POINT; IN CASES
C WHERE THE ALLELE FREQUENCIES SUMMED TO SLIGHTLY LESS THAN 1
C AND WE GOT A RANDOM NUMBER VERY NEAR 1 IN SIMGEN, IT WAS
C POSSIBLE FOR A GENOTYPE TO BE HALF MISSING E.G. "268/   "
C
       GCUT(ILOC,NA)=1.D0
 30    CONTINUE
C
       RETURN
       END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE SIMGEN(GCUT,SGENO,ISEED1,ISEED2,ISEED3,MAXALL
     1,MAXLOC,MAXPEO,MXPTOT,NHAP,NALL,NLOCI,THETA,I,ERROR)
C
C THIS SUBROUTINE SIMULATES MARKER GENOTYPES FOR A 4-PERSON NUCLEAR
C FAMILY, INCLUDING THE INTRODUCTION OF ERROR.
C
C CURRENT PEDIGREE.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DOUBLE PRECISION GCUT(MAXLOC,MAXALL),THETA(MAXLOC)
      INTEGER SGENO(MXPTOT,MAXLOC,NHAP),NALL(MAXLOC)
C
C SIMULATE GENOTYPES FOR THE PARENTS I.E. SIMULATE HAPLOTYPES FOR
C THE PARENTS ASSUMING HARDY-WEINBERG AND LINKAGE EQUILIBRIA.
C
      DO 50 NPAR=1,2
       DO 40 ILOC=1,NLOCI
        NA=NALL(ILOC)
         DO 30 IHAP=1,2
          X=RANDM(ISEED1,ISEED2,ISEED3)
           DO 20 IALL=1,NA
            IF(X.GT.GCUT(ILOC,IALL)) GO TO 20
             SGENO(NPAR,ILOC,IHAP)=IALL
             GO TO 30
 20          CONTINUE
 30      CONTINUE
 40    CONTINUE
 50   CONTINUE
C
C SIMULATE GENOTYPES FOR THE TWO SIBLINGS IN THE PEDIGREE.
C
      DO 80 NSIB=3,4
       DO 70 NPAR=1,2
        IHAP=1
        X=RANDM(ISEED1,ISEED2,ISEED3)
        IF(X.GT.0.5D0)IHAP=2
        SGENO(NSIB,1,NPAR)=SGENO(NPAR,1,IHAP)
C
C FOR EACH SUBSEQUENT LOCUS, SIMULATE POSSIBLE RECOMBINATION, AND
C COMPLETE THE HAPLOTYPE.
C
        DO 60 ILOC=2,NLOCI
         X=RANDM(ISEED1,ISEED2,ISEED3)
         IF(X.LT.THETA(ILOC-1))IHAP=3-IHAP
         SGENO(NSIB,ILOC,NPAR)=SGENO(NPAR,ILOC,IHAP)
 60     CONTINUE
 70    CONTINUE
C
C SIMULATE GENOTYPE ERROR FOR SIB EXCEPT AT MARKER I.
C
      DO 120 ILOC=1,NLOCI
      IF (ILOC.LE.I.AND.ILOC.GE.I)GO TO 120
      X=RANDM(ISEED1,ISEED2,ISEED3)
      NA=NALL(ILOC)
      IF(X.GT.ERROR)GO TO 120 
      DO 110 IHAP=1,2
       X=RANDM(ISEED1,ISEED2,ISEED3)
       DO 100 IALL=1,NA
        IF(X.GT.GCUT(ILOC,IALL))GO TO 100
        SGENO(NSIB,ILOC,IHAP)=IALL
        GO TO 110
 100   CONTINUE
 110  CONTINUE
 120  CONTINUE
C
  80  CONTINUE
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      FUNCTION RANDM(ISEED1,ISEED2,ISEED3)
C
C GENERATE A U(0,1) RANDOM NUMBER USING THE METHOD OF WICHMAN AND HILL.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
      ISEED1 = 171 * MOD(ISEED1,177) - 2 * (ISEED1 / 177)
      ISEED2 = 172 * MOD(ISEED2,176) - 35 * (ISEED2 / 176)
      ISEED3 = 170 * MOD(ISEED3,178) - 63 * (ISEED3 / 178)
C
      IF(ISEED1.LT.0) ISEED1 = ISEED1 + 30269
      IF(ISEED2.LT.0) ISEED2 = ISEED2 + 30307
      IF(ISEED3.LT.0) ISEED3 = ISEED3 + 30323
C
      RANDM=DMOD(ISEED1/30269.0D0+ISEED2/30307.0D0
     1      +ISEED3/30323.0D0,1.0D0)
C
      RETURN
      END
C
C
C-----------------------------------------------------------------------
C
      SUBROUTINE REPORT(CM,DOSIM,ECUT,ID,KPUT,KRES,LOCNAM,MAXLOC,MAXSIB
     1,MXPTOT,NISIB,NJSIB,NLOCI,NPAIR,PCUT,PEDID,PROBCE,PVALUE)
C
C PRODUCE A REPORT SUMMARIZING THE RESULTS.
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER*8 ID(MXPTOT),LOCNAM(MAXLOC),PEDID(MXPTOT)
      DOUBLE PRECISION PROBCE(MAXSIB,MAXLOC),PVALUE(MAXSIB,MAXLOC)
     1,CM(MAXLOC)
      INTEGER NISIB(MAXSIB),NJSIB(MAXSIB)
      LOGICAL DOSIM
C
      CALL DRAWLN(KRES,1)
      WRITE(KRES,101)
  101 FORMAT(' ***SUMMARY OF RESULTS*** ')
      CALL DRAWLN(KRES,1)
C
C REPORT SIBLING PAIRS WHOSE POSTERIOR PROBABILITY (OF INCORRECT
C MARKER DATA AT A PARTICULAR LOCUS GIVEN ALL MARKER DATA AND
C PRIOR ERROR) EXCEEDS THE CUTOFF CONSTANT DETERMINED BY SIMULATION.
C
      IF(DOSIM)THEN
C
      WRITE(KRES,102)
  102 FORMAT(' PAIRS IDENTIFIED WITH LIKELY MARKER GENOTYPING ERRORS ')
      WRITE(KRES,103)
  103 FORMAT(' OR MUTATIONS: ')
C
      CALL DRAWLN(KRES,1) 
      WRITE(KRES,104)
  104 FORMAT(' FAMILY ',3X,'SIB1',6X,'SIB2',6X,'MARKER',5X
     1,'POS(CM)',3X,'PR.ERROR')
      CALL DRAWLN(KRES,1)
C
      NCOUNT=0
C
      DO 200 JPAIR=1,NPAIR
       DO 190 I=1,NLOCI
C
      IF(PVALUE(JPAIR,I).GT.PCUT)GO TO 190
      WRITE(KRES,105) PEDID(NISIB(JPAIR)),ID(NISIB(JPAIR))
     1,ID(NJSIB(JPAIR)),LOCNAM(I),CM(I),1-PROBCE(JPAIR,I)
  105 FORMAT(' ',4(A8,2X),1X,F5.1,6X,F8.6)
C
C WRITE FLAGGED SIB-PAIR-MARKER COMBINATION INFORMATION TO THE
C OUTPUT FILE FOR OPTIONAL POST PROCESSING.
C
      WRITE(KPUT,105) PEDID(NISIB(JPAIR)),ID(NISIB(JPAIR))
     1,ID(NJSIB(JPAIR)),LOCNAM(I),CM(I),1-PROBCE(JPAIR,I)
C
C
      NCOUNT=1+NCOUNT
C
 190  CONTINUE
 200  CONTINUE
C
      CALL DRAWLN(KRES,1)
C
C REPORT SIBLING PAIRS WHOSE POSTERIOR PROBABILITY (OF INCORRECT
C MARKER DATA AT A PARTICULAR LOCUS GIVEN ALL MARKER DATA AND
C PRIOR ERROR) EXCEEDS THE USER DEFINED VALUE.

      ELSE
      WRITE(KRES,110)
 110  FORMAT(' PAIRS IDENTIFIED WITH HIGH POSTERIOR ERROR ')
      WRITE(KRES,111)
 111  FORMAT(' PROBABILITIES: ')
      CALL DRAWLN(KRES,1) 
      WRITE(KRES,104)
      CALL DRAWLN(KRES,1)
C
      NCOUNT=0
C
      DO 220 JPAIR=1,NPAIR
       DO 210 I=1,NLOCI
C
      IF(1-PROBCE(JPAIR,I).LT.ECUT)GO TO 210
      WRITE(KRES,105) PEDID(NISIB(JPAIR)),ID(NISIB(JPAIR))
     1,ID(NJSIB(JPAIR)),LOCNAM(I),CM(I),1-PROBCE(JPAIR,I)
C
C WRITE FLAGGED SIB-PAIR-MARKER COMBINATION INFORMATION TO THE
C OUTPUT FILE FOR OPTIONAL POST PROCESSING.
C
      WRITE(KPUT,105) PEDID(NISIB(JPAIR)),ID(NISIB(JPAIR))
     1,ID(NJSIB(JPAIR)),LOCNAM(I),CM(I),1-PROBCE(JPAIR,I)
C
      NCOUNT=1+NCOUNT
C
 210  CONTINUE
 220  CONTINUE
C
      CALL DRAWLN(KRES,1)
C
      END IF
C
C REPORT SHORT SUMMARY OF TOTALS.
C
      WRITE(KRES,106)NCOUNT
 106  FORMAT(' SIBMED IDENTIFIED',I4,' SIB-PAIR-MARKER COMBINATIONS')
C
      CALL DRAWLN(KRES,1)
      WRITE(KRES,108)NPAIR
 108  FORMAT(I5,' SIB PAIRS ANALYZED ')
      WRITE(KRES,109)NPAIR*NLOCI
 109  FORMAT(I8,' SIB-PAIR-MARKER COMBINATIONS CONSIDERED')
C
      CALL DRAWLN(KRES,1)
C
      RETURN
      END

